/* (C) Copyright jb2works.com (c) 2004 */
package com.jb2works.reference;

import junit.framework.TestCase;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.StringTokenizer;

/**
 * Extends TestCase to test for strongly reachable instances by specifying
 * the instance itself or the its Class.
 * Strongly reachable instances are mostly causes
 * of Memory Leaks in Java.
 *
 * @author Joerg Baumgaertel
 * @version $Revision: 1.3 $ $Date: 2004/01/14 17:16:08 $
 */
public class MemoryTestCase extends TestCase
{
    public MemoryTestCase(String s) {
        super(s);
    }

    /**
     * assert one or more instances of the specified Class are strongly reachable
     *
     * @param message specify a text that you want to see in the AssertionFailedError
     * */
    public void assertStronglyReachable(String message, Class clazz) {
        // if not strongly reachable, we do not have a dump
        // if strongly reachable, we dont need the dump
        final boolean doDump = false;
        final Object main = this;
        assertTrue0( message, Scanner.stronglyReachable( clazz, main, doDump ) );
    }
    /* assert one or more instances of the specified Class are strongly reachable */
    public void assertStronglyReachable(Class clazz) {
        assertStronglyReachable( null, clazz);
    }

    /**
     * assert no instance of the specified Class is strongly reachable
     *
     * @param message specify a text that you want to see in the AssertionFailedError
     * */
    public void assertNotStronglyReachable(String message, Class clazz) {
        // if strongly reachable, we want a dump on system.err
        final boolean doDump = true;
        final Object main = this;
        assertTrue0( message, ! Scanner.stronglyReachable( clazz, main, doDump ));
    }
    /** assert no instance of the specified Class is strongly reachable */
    public void assertNotStronglyReachable( Class clazz) {
        assertNotStronglyReachable( null, clazz);
    }

    /**
     * assert the specified instance is strongly reachable
     *
     * @param message for the AssertionFailedError
     * */
    public void assertStronglyReachable(String message, Object instance) {
        // if not strongly reachable, we do not have a dump
        // if strongly reachable, we dont need the dump
        final boolean doDump = false;
        final Object main = this;
        assertTrue0( message, Scanner.stronglyReachable( instance, main, doDump ));
    }
    /** assert the specified instance is strongly reachable */
    public void assertStronglyReachable(Object instance) {
        assertStronglyReachable( null, instance );
    }

    /**
     * assert the specified instance is not strongly reachable
     *
     * @param message for the AssertionFailedError
     * */
    public void assertNotStronglyReachable(String message, Object instance) {
        // if strongly reachable, we want a dump on system.err
        final boolean doDump = true;
        final Object main = this;
        assertTrue0( message, ! Scanner.stronglyReachable( instance, main, doDump ));
    }
    /**  assert the specified instance is not strongly reachable */
    public void assertNotStronglyReachable(Object instance) {
        assertNotStronglyReachable( null, instance );
    }

    private static void assertTrue0( String message, boolean condition) {
        if(!condition) {
            if(message != null) {
                throw new AssertionFailedError(message);
            } else {
                throw new AssertionFailedError();
            }
        }
    }

    /** filters the stacktrace from internal testcase functions */
    private static final class AssertionFailedError extends junit.framework.AssertionFailedError {
        
        private AssertionFailedError(String message) {
            super( message );
        }
        private AssertionFailedError() {
        }

        // BookSeller example output without printStackTrace() filter:
        //
        // There was 1 failure:
        // 1) testBookSeller(com.jb2works.reference.BookSellerTest)junit.framework.AssertionFailedError
        //    at com.jb2works.reference.MemoryTestCase.assertNotStronglyReachable(MemoryTestCase.java:38)
        //    at com.jb2works.reference.MemoryTestCase.assertNotStronglyReachable(MemoryTestCase.java:41)
        //    at com.jb2works.reference.BookSellerTest.testBookSeller(BookSellerTest.java:54)
        //    at sun.reflect.NativeMethodAccessorImpl.invoke0(Native Method)
        //    at sun.reflect.NativeMethodAccessorImpl.invoke(NativeMethodAccessorImpl.java:39)
        //    at sun.reflect.DelegatingMethodAccessorImpl.invoke(DelegatingMethodAccessorImpl.java:25)
        //    ..

        /** 
         * filters the "com.jb2works.reference.MemoryTestCase." calls from the StackTrace.
         * users want to be set on the exactly point of assertion failures 
         * */
        public void printStackTrace( PrintWriter ps ) {
            final StringWriter stringWriter = new StringWriter();
            super.printStackTrace( new PrintWriter( stringWriter ));            
            final StringTokenizer tokenizer = new StringTokenizer( stringWriter.toString(), "\n");
            while(tokenizer.hasMoreTokens()) {
                final String next = tokenizer.nextToken();
                // filter lines containing MemoryTestCase but not MemoryTestCase.AssertionFailedError
                if(next.indexOf( MemoryTestCase.class.getName() )==-1 || 
                   next.indexOf( MemoryTestCase.AssertionFailedError.class.getName() )>-1) {
                    ps.println( next );
                }
            }
        }
    }
}
