package ciss.phase_viewer.plugins.viewer.edit;

import java.awt.BasicStroke;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.text.DecimalFormat;
import java.util.Vector;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.border.TitledBorder;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;

import org.apache.log4j.Logger;

import ciss.phase_viewer.acviewer.ACVAction;
import ciss.phase_viewer.acviewer.J3DPanel;
import ciss.phase_viewer.acviewer.MainPanel;
import ciss.phase_viewer.atomcoord.Atom;
import ciss.phase_viewer.atomcoord.AtomCoords;
import ciss.phase_viewer.atomcoord.AtomList;
import ciss.phase_viewer.atomcoord.Cell;
import ciss.phase_viewer.common.VectorOperations;
import ciss.phase_viewer.mainpanel.Desk;
import ciss.phase_viewer.mainpanel.InternalFrameChase;
import ciss.phase_viewer.settings.GlobalProperties;
import ciss.phase_viewer.settings.PropertiesManager;

public class ABaxisAction extends ACVAction {

	public ABaxisAction(J3DPanel j3dPanel) {
		super(j3dPanel);
	}

	@Override
	public void actionPerformed(ActionEvent e) {
		new ABaxisPanel((MainPanel) parent);
	}

}

interface A1A2Setter {
	public void setA1A2(int[] a1, int[] a2);
}

class ABaxisPanel extends InternalFrameChase implements A1A2Setter {
	private Logger logger = Logger.getLogger(ABaxisPanel.class.getName());
	private MainPanel parent;
	private AtomCoords orgCoords;

	public ABaxisPanel(MainPanel panel) {
		super("ab-axis", new java.awt.Dimension(800, 600));

		this.parent = panel;
		this.parent.addDisposeOnExit(this);
		this.orgCoords = this.parent.getCD().getAtomCoords().getCopy();
		if (this.parent.getCD() == null) {
			dispose();
			JOptionPane.showInternalMessageDialog(Desk.getDesktop().getSelectedFrame(),
					"Atomic coordinates not allocated");
			logger.error("Atomic coordinates not allocated");
			return;
		}
		init();
	}

	private JTextField tfa1a;
	private JTextField tfa1b;
	private JTextField tfa2a;
	private JTextField tfa2b;
	private LatticePointsPanel latticePointsPanel;
	private DecimalFormat form = new DecimalFormat("0.000");
	private JLabel a1Label;
	private JLabel a2Label;
	private CaretListener caretListener;
	private JCheckBox cbParallel;
	private JCheckBox cbontheFly;

	private void init() {
		JPanel jp = new JPanel();

		tfa1a = new JTextField("1");
		tfa1b = new JTextField("0");
		tfa2a = new JTextField("0");
		tfa2b = new JTextField("1");
		tfa1a.setColumns(20);
		tfa1b.setColumns(20);
		tfa1a.setHorizontalAlignment(JTextField.CENTER);
		tfa1b.setHorizontalAlignment(JTextField.CENTER);
		tfa1a.setSize(tfa1a.getSize().width, 80);
		tfa1b.setSize(tfa1a.getSize().width, 80);
		tfa2a.setColumns(20);
		tfa2b.setColumns(20);
		tfa2a.setHorizontalAlignment(JTextField.CENTER);
		tfa2b.setHorizontalAlignment(JTextField.CENTER);
		tfa2a.setSize(tfa1a.getSize().width, 80);
		tfa2b.setSize(tfa1a.getSize().width, 80);
		JPanel pa1a = new JPanel();
		JPanel pa1b = new JPanel();
		//		pa1a.setLayout(new BoxLayout(pa1a, BoxLayout.X_AXIS));
		//		pa1b.setLayout(new BoxLayout(pa1b, BoxLayout.X_AXIS));

		pa1a.setBorder(new TitledBorder("a"));
		pa1b.setBorder(new TitledBorder("b"));
		JPanel pa2a = new JPanel();
		JPanel pa2b = new JPanel();
		pa2a.setBorder(new TitledBorder("a"));
		pa2b.setBorder(new TitledBorder("b"));
		pa1a.add(tfa1a);
		pa1b.add(tfa1b);
		pa2a.add(tfa2a);
		pa2b.add(tfa2b);

		a1Label = new JLabel("a1");
		a2Label = new JLabel("a2");

		JPanel a1 = new JPanel();
		JPanel a2 = new JPanel();
		a1.setLayout(new BoxLayout(a1, BoxLayout.X_AXIS));
		a1.add(a1Label);
		a1.add(pa1a);
		a1.add(pa1b);
		a1.setPreferredSize(new Dimension(300, 60));

		a2.setLayout(new BoxLayout(a2, BoxLayout.X_AXIS));
		a2.add(a2Label);
		a2.add(pa2a);
		a2.add(pa2b);
		a2.setPreferredSize(new Dimension(300, 60));

		JPanel pcheck = new JPanel();
		cbParallel = new JCheckBox("make the a1 axis parallel to the x-axis");
		cbParallel.setSelected(true);
		cbontheFly = new JCheckBox("on the fly");
		pcheck.add(cbParallel);
		pcheck.add(cbontheFly);
		jp.setLayout(new BoxLayout(jp, BoxLayout.Y_AXIS));
		jp.add(a1);
		jp.add(a2);
		jp.add(pcheck);

		JButton bapply = new JButton("apply");
		JButton bclose = new JButton("close");

		JPanel pbtn = new JPanel();
		//		pbtn.setLayout(new BoxLayout(pbtn, BoxLayout.X_AXIS));
		pbtn.add(bapply);
		pbtn.add(bclose);

		JPanel pp = new JPanel();
		pp.setLayout(new BorderLayout());
		//		jp.setSize(new Dimension(jp.getSize().width,200));
		pp.add(jp, BorderLayout.NORTH);
		latticePointsPanel = new LatticePointsPanel(parent.getCD().getAtomCoords(), this);
		JScrollPane scrPane = new JScrollPane(latticePointsPanel);
		pp.add(scrPane, BorderLayout.CENTER);
		pp.add(pbtn, BorderLayout.SOUTH);
		getContentPane().add(pp);

		bclose.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				dispose();
			}
		});

		bapply.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				doIt();
			}
		});
		cbontheFly.addActionListener(new ActionListener() {

			@Override
			public void actionPerformed(ActionEvent e) {
				if (cbontheFly.isSelected())
					doIt();
			}
		});

		caretListener = new CaretListener() {
			@Override
			public void caretUpdate(CaretEvent e) {
				int[] a1 = new int[2];
				int[] a2 = new int[2];
				try {
					a1[0] = Integer.parseInt(tfa1a.getText().trim());
					a1[1] = Integer.parseInt(tfa1b.getText().trim());
					a2[0] = Integer.parseInt(tfa2a.getText().trim());
					a2[1] = Integer.parseInt(tfa2b.getText().trim());
				} catch (NumberFormatException nfe) {
					logger.error("an integer must be provided");
					return;
				}
				latticePointsPanel.setA1A2(a1, a2);
				setA1A2Label();
			}
		};
		tfa1a.addCaretListener(caretListener);
		tfa1b.addCaretListener(caretListener);
		tfa2a.addCaretListener(caretListener);
		tfa2b.addCaretListener(caretListener);

		setA1A2Label();

	}

	private double[][] getCellAB() {
		double[][] ret = { { 0d, 0d }, { 0d, 0d } };
		int a1a;
		int a1b;
		int a2a;
		int a2b;
		try {
			a1a = Integer.parseInt(tfa1a.getText().trim());
			a1b = Integer.parseInt(tfa1b.getText().trim());
			a2a = Integer.parseInt(tfa2a.getText().trim());
			a2b = Integer.parseInt(tfa2b.getText().trim());
		} catch (NumberFormatException nfe) {
			return ret;
		}
		AC ac = extendCellAndRotate(a1a, a1b, a2a, a2b, false);
		if (ac == null)
			return null;
		double[][] lat = ac.getLatVec();
		for (int i = 0; i < 2; i++)
			for (int j = 0; j < 2; j++)
				ret[i][j] = lat[i][j];
		return ret;
	}

	private void doIt() {
		int a1a;
		int a1b;
		int a2a;
		int a2b;
		try {
			a1a = Integer.parseInt(tfa1a.getText().trim());
			a1b = Integer.parseInt(tfa1b.getText().trim());
			a2a = Integer.parseInt(tfa2a.getText().trim());
			a2b = Integer.parseInt(tfa2b.getText().trim());
		} catch (NumberFormatException nfe) {
			err("the numbers must be integers");
			return;
		}
		if (a1a == 0 && a1b == 0) {
			err("a1a and a1b cannot be zero at the same time");
			return;
		}
		if (a2a == 0 && a2b == 0) {
			err("a2a and a2b cannot be zero at the same time");
			return;
		}
		extendCellAndRotate(a1a, a1b, a2a, a2b);
	}

	private boolean isParallel(int a1a, int a1b, int a2a, int a2b) {
		if (a1b == 0 && a2b == 0 || a1a == 0 && a2a == 0)
			return true;
		if (a1a != 0 && a2a != 0)
			return Math.abs((double) a1b / (double) a1a - (double) a2b / (double) a2a) < 1e-12;
		return false;
	}

	private AC extendCellAndRotate(int a1a, int a1b, int a2a, int a2b) {
		return extendCellAndRotate(a1a, a1b, a2a, a2b, true);

	}

	private AC extendCellAndRotate(int a1a, int a1b, int a2a, int a2b, boolean update) {
		AtomCoords coords = orgCoords.getCopy();
		int orgnatm = coords.getNumAt();
		double[][] latvec = coords.getCellDouble(Cell.CELL_VEC);
		if (latvec == null) {
			if (update)
				err("cellvector is not allocated");
			return null;
		}
		if (a1a == a2a && a1b == a2b) {
			if (update)
				err("a1 and a2 must differ");
			return null;
		}
		if (isParallel(a1a, a1b, a2a, a2b)) {
			if (update)
				err("a1 and a2 must not be parallel");
			return null;
		}

		double[][] cpos = coords.getPosDouble();
		String[] elem = coords.getElements();
		AC ac = new AC(elem, cpos, latvec);

		double[] avec = latvec[0];
		double[] bvec = latvec[1];

		double[] newavec = new double[3];
		double[] newbvec = new double[3];
		for (int i = 0; i < 3; i++) {
			newavec[i] = avec[i] * a1a + bvec[i] * a1b;
			newbvec[i] = avec[i] * a2a + bvec[i] * a2b;
		}
		int supsizea = Math.abs(a1a) + Math.abs(a2a);
		int supsizeb = Math.abs(a1b) + Math.abs(a2b);
		ac.toSuper(supsizea, supsizeb, 1);
		double[][] newlatvec = new double[3][3];
		newlatvec[0] = newavec;
		newlatvec[1] = newbvec;
		newlatvec[2] = latvec[2];
		ac.setLatVec(newlatvec);

		logger.debug("after new cell " + ac.toString());
		if (cbParallel.isSelected()) {
			double rotang = Math.asin(newavec[1] / Math.sqrt(newavec[0] * newavec[0] + newavec[1] * newavec[1]));
			ac.rotateXY(rotang);
		}

		logger.debug("after rotation " + ac.toString());
		ac.removeDuplicate();
		ac.pack();
		logger.debug("after removing duplicates " + ac.toString());
		latvec = ac.getLatVec();
		cpos = ac.getCartPos();
		String[] ename = ac.getElem();

		if (update) {
			AtomCoords actmp = parent.getCD().getAtomCoords();
			actmp.saveState();
			AtomList alist = actmp.getAtomList();
			int icount = 0;
			for (int i = alist.getNumAt() - 1; i >= 0; i--) {
				alist.removeAtomAt(i);
				icount++;
			}

			Cell cell = new Cell(latvec[0], latvec[1], latvec[2]);
			actmp.setCell(cell);
			//			actmp.convertCell(Cell.CELL_ABC);
			for (int i = 0; i < cpos.length; i++) {
				Atom at = new Atom(ename[i], cpos[i]);
				actmp.getAtomList().addAtom(at, true);
				icount++;
			}
			actmp.getAtomList().setUndoCount(icount+1);
			parent.getCD().setCoords(actmp);
			actmp.finalizeState();
		}
		return ac;
	}

	private void err(String errs) {
		JOptionPane.showInternalMessageDialog(Desk.getDesktop().getSelectedFrame(),
				errs);
		logger.error(errs);
	}

	@Override
	public void setA1A2(int[] a1, int[] a2) {
		boolean theSame = true;
		try {
			int ii = Integer.parseInt(tfa1a.getText());
			if (a1[0] != ii)
				theSame = false;
			ii = Integer.parseInt(tfa1b.getText());
			if (a1[1] != ii)
				theSame = false;
			ii = Integer.parseInt(tfa2a.getText());
			if (a2[0] != ii)
				theSame = false;
			ii = Integer.parseInt(tfa2b.getText());
			if (a2[1] != ii)
				theSame = false;
		} catch (NumberFormatException e) {
			return;
		}
		if (theSame)
			return;
		setA1A2(tfa1a, a1[0]);
		setA1A2(tfa1b, a1[1]);
		setA1A2(tfa2a, a2[0]);
		setA1A2(tfa2b, a2[1]);
		boolean b = setA1A2Label();
		if (b && cbontheFly.isSelected())
			extendCellAndRotate(a1[0], a1[1], a2[0], a2[1]);
	}

	private void setA1A2(JTextField tf, int a) {
		tf.removeCaretListener(caretListener);
		tf.setText(String.valueOf(a));
		tf.addCaretListener(caretListener);

	}

	private boolean setA1A2Label() {
		double[][] lat = getCellAB();
		if (lat == null) {
			err("invalid lattice");
			return false;
		}
		logger.debug("new avec " + lat[0][0] + " " + lat[0][1]);
		logger.debug("new bvec " + lat[1][0] + " " + lat[1][1]);
		a1Label.setText("a1 (" + form.format(lat[0][0]) + ", " + form.format(lat[0][1]) + ")");
		a2Label.setText("a2 (" + form.format(lat[1][0]) + ", " + form.format(lat[1][1]) + ")");
		return true;
	}

}

class AC implements Cloneable {
	private Logger logger = Logger.getLogger(AC.class.getName());
	private int natm;
	private String[] elem;
	private double[][] cpos;
	private double[][] fpos;
	private double[][] latvec;
	private double[][] invlat;
	private double volume;

	public Object clone() {
		return new AC(elem.clone(), cpos.clone(), latvec.clone());
	}

	AC(String[] elem, double[][] cpos, double[][] latvec) {
		this.elem = elem;
		this.cpos = cpos;
		this.natm = this.cpos.length;
		this.latvec = latvec;
		this.setVolume();
		this.setInvLat();

		this.toFract();
	}

	double[][] getCartPos() {
		return this.cpos;
	}

	double[][] getFractPos() {
		return this.fpos;
	}

	void rotateXY(double rotang) {
		double[][] rotmat = new double[2][2];
		rotmat[0][0] = Math.cos(rotang);
		rotmat[0][1] = Math.sin(rotang);
		rotmat[1][0] = -Math.sin(rotang);
		rotmat[1][1] = Math.cos(rotang);
		double[] avec = this.latvec[0];
		avec = rotateSub(avec, rotmat);
		for (int i = 0; i < 2; i++)
			if (Math.abs(avec[i]) < 1e-10)
				this.latvec[0][i] = 0;
			else
				this.latvec[0][i] = avec[i];
		double[] bvec = this.latvec[1];
		bvec = rotateSub(bvec, rotmat);
		for (int i = 0; i < 2; i++)
			if (Math.abs(bvec[i]) < 1e-10)
				this.latvec[1][i] = 0;
			else
				this.latvec[1][i] = bvec[i];
		setInvLat();
		for (int i = 0; i < this.natm; i++) {
			double[] xy = rotateSub(cpos[i], rotmat);
			for (int j = 0; j < 2; j++)
				cpos[i][j] = xy[j];
		}
		toFract();
	}

	int getNumAt() {
		return cpos.length;
	}

	void setLatVec(double[][] latvec) {
		this.latvec = latvec;
		setVolume();
		setInvLat();
		toFract();
	}

	double[][] getLatVec() {
		return this.latvec;
	}

	String[] getElem() {
		return this.elem;
	}

	void pack() {
		for (int i = 0; i < natm; i++)
			for (int j = 0; j < 3; j++) {
				if (this.fpos[i][j] > 1)
					this.fpos[i][j] = this.fpos[i][j] - Math.floor(this.fpos[i][j]);
				else if (this.fpos[i][j] < 0)
					this.fpos[i][j] = this.fpos[i][j] + Math.ceil(-this.fpos[i][j]);
				if (Math.abs(Math.abs(fpos[i][j]) - 1) < 1e-10)
					fpos[i][j] = 0;
			}
		this.toCart();
	}

	private double[] rotateSub(double[] source, double[][] rotmat) {
		double[] ret = new double[2];
		for (int i = 0; i < 2; i++) {
			ret[i] = 0.d;
			for (int j = 0; j < 2; j++)
				ret[i] += rotmat[i][j] * source[j];
		}
		return ret;
	}

	void toFract() {
		if (invlat == null) {
			logger.error("invalid cell!!");
			return;
		}
		this.fpos = new double[natm][3];
		for (int i = 0; i < natm; i++)
			for (int j = 0; j < 3; j++) {
				this.fpos[i][j] = 0d;
				for (int k = 0; k < 3; k++)
					this.fpos[i][j] += this.cpos[i][k] * this.invlat[k][j];
			}
	}

	void toCart() {
		this.cpos = new double[natm][3];
		for (int i = 0; i < natm; i++)
			for (int j = 0; j < 3; j++) {
				this.cpos[i][j] = 0d;
				for (int k = 0; k < 3; k++)
					this.cpos[i][j] += this.fpos[i][k] * this.latvec[k][j];
			}
	}

	void toSuper(int na, int nb, int nc) {
		int newnatm = na * nb * nc * natm;
		double[][] newcpos = new double[newnatm][3];
		String[] newelem = new String[newnatm];
		int count = 0;
		for (int i = 0; i < na; i++)
			for (int j = 0; j < nb; j++)
				for (int k = 0; k < nc; k++) {
					double[] trans = new double[3];
					for (int l = 0; l < 3; l++)
						trans[l] = latvec[0][l] * i + latvec[1][l] * j + latvec[2][l] * k;
					for (int l = 0; l < natm; l++) {
						for (int m = 0; m < 3; m++)
							newcpos[count][m] = cpos[l][m] + trans[m];
						newelem[count] = String.valueOf(elem[l]);
						count += 1;
					}
				}
		for (int i = 0; i < 3; i++) {
			this.latvec[0][i] = this.latvec[0][i] * na;
			this.latvec[1][i] = this.latvec[1][i] * nb;
			this.latvec[2][i] = this.latvec[2][i] * nc;
		}
		natm = newnatm;
		this.setVolume();
		this.setInvLat();
		cpos = newcpos;
		elem = newelem;
		this.toFract();
	}

	void removeDuplicate() {
		removeDuplicate(1e-10);
	}

	void removeDuplicate(double eps) {
		pack();
		Vector<Integer> remvec = new Vector<Integer>();
		for (int i = 0; i < fpos.length - 1; i++) {
			for (int j = i + 1; j < fpos.length; j++) {
				double dj = 0d;
				for (int k = 0; k < 3; k++) {
					dj += Math.pow((fpos[j][k] - fpos[i][k]), 2);
				}
				if (Math.abs(dj) < eps && remvec.indexOf(j) < 0)
					remvec.add(j);
			}
		}
		int newnatm = natm - remvec.size();
		double[][] newfpos = new double[newnatm][3];
		int icount = 0;
		for (int i = 0; i < natm; i++)
			if (remvec.indexOf(i) < 0) {
				newfpos[icount] = fpos[i];
				icount++;
			}
		natm = newnatm;
		fpos = newfpos;
		toCart();
	}

	private void setVolume() {
		this.volume = VectorOperations.dotProduct(latvec[0], VectorOperations.crossProduct(latvec[1], latvec[2]));
	}

	private void setInvLat() {
		this.invlat = VectorOperations.getInverseMatrix(latvec);
	}

	public String toString() {
		String ret = "lattice\n";

		for (int i = 0; i < 3; i++)
			ret += latvec[i][0] + " " + latvec[i][1] + " " + latvec[i][2] + "\n";
		if (invlat != null) {
			ret += "invlat\n";
			for (int i = 0; i < 3; i++)
				ret += invlat[i][0] + " " + invlat[i][1] + " " + invlat[i][2] + "\n";
		}

		ret += "cart pos\n";
		for (int i = 0; i < natm; i++)
			ret += String.valueOf(i) + " " + cpos[i][0] + " " + cpos[i][1] + " " + cpos[i][2] + "\n";
		ret += "fract pos\n";
		for (int i = 0; i < natm; i++)
			ret += String.valueOf(i) + " " + fpos[i][0] + " " + fpos[i][1] + " " + fpos[i][2] + "\n";
		return ret;

	}
}

class LatticePointsPanel extends JPanel implements MouseListener, A1A2Setter {
	enum POINT_SELECTION_STATE {
		NOT_IN_SELECTION, SELECTING_A1, SELECTING_A2
	}

	private Logger logger = Logger.getLogger(LatticePointsPanel.class.getName());
	private AtomCoords coords;
	private GlobalProperties props = PropertiesManager.getGlobalProperties(PropertiesManager.PROPERTIES_ACV, true);
	private int maxA = props.getIntProperty("latticepanel_maxA", 15);
	private int maxB = props.getIntProperty("latticepanel_maxB", 15);
	private int maxAh = maxA / 2;
	private int maxBh = maxB / 2;
	private int factor = props.getIntProperty("latticepanel_interval", 50);
	private int rad = props.getIntProperty("latticepanel_radius", 10);
	private int lineWidth = props.getIntProperty("latticepanel_linewidth", 5);
	private Color a1Color = new Color(props.getIntProperty("latticepanel_a1ColorR", 0),
			props.getIntProperty("latticepanel_a1ColorG", 0), props.getIntProperty("latticepanel_a1ColorB", 255));
	private Color a2Color = new Color(props.getIntProperty("latticepanel_a2ColorR", 0),
			props.getIntProperty("latticepanel_a2ColorG", 174), props.getIntProperty("latticepanel_a2ColorB", 239));
	private Color oColor = new Color(props.getIntProperty("latticepanel_oColorR", 255),
			props.getIntProperty("latticepanel_oColorG", 0), props.getIntProperty("latticepanel_oColorB", 0));
	private Color pColor = new Color(props.getIntProperty("latticepanel_pColorR", 0),
			props.getIntProperty("latticepanel_pColorG", 0), props.getIntProperty("latticepanel_pColorB", 0));
	private int[] currAPos = { 1, 0 };
	private int[] currBPos = { 0, 1 };
	private int[] xyoff = props.getIntArrayProperty("latticepanel_xyoff", new int[] { 50, 50 });
	private Ellipse2D[][] points = new Ellipse2D[maxAh * 2 + 1][maxBh * 2 + 1];
	private POINT_SELECTION_STATE state = POINT_SELECTION_STATE.NOT_IN_SELECTION;
	private A1A2Setter a1a2setter;
	private boolean fromListener = false;
	private double[][] lat;
	private JTextField tfmaxa;
	private JTextField tfmaxb;

	LatticePointsPanel(AtomCoords coords, A1A2Setter a1a2setter) {
		super();
		this.coords = coords;
		this.a1a2setter = a1a2setter;
		this.lat = coords.getCellDouble(Cell.CELL_VEC).clone();
		tfmaxa = new JTextField(10);
		tfmaxa.setHorizontalAlignment(JTextField.CENTER);
		tfmaxa.setText(String.valueOf(maxA));
		tfmaxb = new JTextField(10);
		tfmaxb.setHorizontalAlignment(JTextField.CENTER);
		tfmaxb.setText(String.valueOf(maxB));
		add(new JLabel("max a"));
		add(tfmaxa);
		add(new JLabel("max b"));
		add(tfmaxb);
		addMouseListener(this);

		CaretListener cl = new CaretListener() {
			@Override
			public void caretUpdate(CaretEvent e) {
				int newMaxA = 0;
				int newMaxB = 0;
				try {
					newMaxA = Integer.parseInt(tfmaxa.getText().trim());
					newMaxB = Integer.parseInt(tfmaxb.getText().trim());
				} catch (NumberFormatException nfe) {
				}
				if (newMaxA < 2 || newMaxB < 2) {
					return;
				}
				if (maxA == newMaxA && maxB == newMaxB)
					return;
				maxA = newMaxA;
				maxB = newMaxB;
				maxAh = maxA / 2;
				maxBh = maxB / 2;
				points = new Ellipse2D[maxAh * 2 + 1][maxBh * 2 + 1];
				update(getGraphics());
			}
		};
		tfmaxa.addCaretListener(cl);
		tfmaxb.addCaretListener(cl);
	}

	private double[][] getNormalizedABLattce() {
		double[][] ret = new double[2][2];
		double longest = 0d;
		for (int i = 0; i < 2; i++) {
			double len = 0d;
			for (int j = 0; j < 3; j++)
				len += lat[i][j] * lat[i][j];
			len = Math.sqrt(len);
			if (len > longest)
				longest = len;
		}
		for (int i = 0; i < 2; i++)
			for (int j = 0; j < 2; j++)
				ret[i][j] = lat[i][j] / longest;
		return ret;
	}

	public void paintComponent(Graphics g) {
		super.paintComponent(g);
		Graphics2D g2d = (Graphics2D) g;

		double[][] lat = getNormalizedABLattce();

		int[] minval = getMinXY(lat);

		Paint oPaint = g2d.getPaint();
		int[] origin = getXYCoordinates(0, 0, lat);
		int[] apos = getXYCoordinates(currAPos[0], -currAPos[1], lat);
		int radh = rad / 2;
		g2d.setStroke(new BasicStroke(lineWidth));
		if (state != POINT_SELECTION_STATE.SELECTING_A1) {
			g2d.setPaint(a1Color);
			g2d.draw(new Line2D.Double(origin[0] + radh - minval[0], origin[1] + radh - minval[1],
					apos[0] + radh - minval[0], apos[1] + radh - minval[1]));
		}
		if (state != POINT_SELECTION_STATE.SELECTING_A2) {
			apos = getXYCoordinates(currBPos[0], -currBPos[1], lat);
			g2d.setPaint(a2Color);
			g2d.draw(new Line2D.Double(origin[0] + radh - minval[0], origin[1] + radh - minval[1],
					apos[0] + radh - minval[0], apos[1] + radh - minval[1]));
		}
		g2d.setPaint(oPaint);
		int maxx = 0;
		int maxy = 0;
		for (int i = -maxAh; i <= maxAh; i++)
			for (int j = -maxBh; j <= maxBh; j++) {
				int[] xy = getXYCoordinates(i, j, lat);
				int x = xy[0] - minval[0];
				int y = xy[1] - minval[1];
				if (x > maxx)
					maxx = x;
				if (y > maxy)
					maxy = y;
				Color color = null;
				if (i == 0 && j == 0)
					color = oColor;
				else if (i == currAPos[0] && j == -currAPos[1] && state != POINT_SELECTION_STATE.SELECTING_A1)
					color = a1Color;
				else if (i == currBPos[0] && j == -currBPos[1] && state != POINT_SELECTION_STATE.SELECTING_A2)
					color = a2Color;
				else
					color = pColor;
				g2d.setPaint(color);
				Ellipse2D e2d = new Ellipse2D.Double(x, y, rad, rad);
				points[i + maxAh][-j + maxBh] = e2d;
				g2d.draw(e2d);

				g2d.fill(e2d);
				g2d.setPaint(oPaint);
			}
		setPreferredSize(new Dimension(maxx + xyoff[0], maxy + xyoff[1]));
		if (state == POINT_SELECTION_STATE.NOT_IN_SELECTION && !fromListener)
			a1a2setter.setA1A2(currAPos, currBPos);

	}

	private int[] getMinXY(double[][] lat) {
		int[] minval = { 10000000, 10000000 };
		for (int i = -maxAh; i <= maxAh; i++)
			for (int j = -maxBh; j <= maxBh; j++) {
				for (int k = 0; k < 2; k++) {
					int[] xy = getXYCoordinates(i, j, lat);
					if (xy[k] < minval[k])
						minval[k] = xy[k];
				}
			}
		for (int i = 0; i < 2; i++)
			minval[i] -= xyoff[i];
		return minval;
	}

	private int[] getXYCoordinates(int i, int j, double[][] lat) {
		int yoff = maxB * factor / 2;
		int xoff = maxA * factor / 2;
		int x = (int) (factor * (i * lat[0][0] - j * lat[1][0])) + xoff;
		int y = -(int) (factor * (i * lat[0][1] - j * lat[1][1])) + yoff;
		return new int[] { x, y };
	}

	@Override
	public void mouseClicked(MouseEvent e) {
		processClick(e);
	}

	@Override
	public void mousePressed(MouseEvent e) {
	}

	@Override
	public void mouseReleased(MouseEvent e) {
	}

	@Override
	public void mouseEntered(MouseEvent e) {
	}

	@Override
	public void mouseExited(MouseEvent e) {
	}

	public void setA1A2(int[] a1, int[] a2) {
		this.currAPos = a1;
		this.currBPos = a2;
		state = POINT_SELECTION_STATE.NOT_IN_SELECTION;
		fromListener = true;
		update(getGraphics());
		revalidate();
		fromListener = false;
	}

	private void processClick(MouseEvent e) {
		boolean doneSomething = false;
		logger.debug("state " + state);
		if (state == POINT_SELECTION_STATE.NOT_IN_SELECTION) {
			logger.debug("not in selection");
			if (points[currAPos[0] + maxAh][currAPos[1] + maxBh].contains(e.getX(), e.getY())) {
				state = POINT_SELECTION_STATE.SELECTING_A1;
				doneSomething = true;
			}
			if (points[currBPos[0] + maxAh][currBPos[1] + maxBh].contains(e.getX(), e.getY())) {
				state = POINT_SELECTION_STATE.SELECTING_A2;
				doneSomething = true;
			}
		} else if (state == POINT_SELECTION_STATE.SELECTING_A1) {
			logger.debug("select a1");
			for (int i = -maxAh; i <= maxAh; i++)
				for (int j = -maxBh; j <= maxBh; j++) {
					if ((i == 0 && j == 0) || (i == currBPos[0] && j == -currBPos[1])) {
						logger.debug("invalid point for a1");
						continue;
					}
					if (points[i + maxAh][-j + maxBh].contains(e.getX(), e.getY())) {
						currAPos[0] = i;
						currAPos[1] = -j;
						state = POINT_SELECTION_STATE.NOT_IN_SELECTION;
						doneSomething = true;
					}
				}
		} else if (state == POINT_SELECTION_STATE.SELECTING_A2) {
			logger.debug("select a2");
			for (int i = -maxAh; i <= maxAh; i++)
				for (int j = -maxBh; j <= maxBh; j++) {
					if ((i == 0 && j == 0) || (i == currAPos[0] && j == -currAPos[1])) {
						logger.debug("invalid point for a2");
						continue;
					}
					if (points[i + maxAh][-j + maxBh].contains(e.getX(), e.getY())) {
						currBPos[0] = i;
						currBPos[1] = -j;
						state = POINT_SELECTION_STATE.NOT_IN_SELECTION;
						doneSomething = true;
					}
				}
		}
		if (doneSomething) {
			update(getGraphics());
			revalidate();
		}
	}
}
