/********************************************************************************
!------------------------------------------------------------------------------
!  SOFTWARE NAME : ppconv version 1.0
!
!  SOURCE FILE NAME: ppconv10.c
!  FUNCTIONS: strupper, constout, strlower, getlistno, Dtolf, atod,
!             dtoa, gotonextline, skip_comment, skip_space, skip_k2k,
!             get_keyword, get_parastr, get_kakko_l, get_kakko_r,
!             get_komma, get_mesh, get_title, get_spin, get_ks_equation,
!             get_configuration, get_core_potential, get_xc_potential,
!             get_mix, get_anderson, get_mixing, get_energy, get_potential,
!             get_max_iteration, get_conv_scf, get_conv_wf, get_convergence,
!             get_predictor_corrector, get_diff_order, get_num_coef_veff,
!             get_output_para, get_ae_output, get_all_electron, get_shape,
!             get_spdf, get_nc, get_us, get_wf, get_local, get_qps_rcut,
!             get_qps, get_cc_rcut, get_core_corre, get_fourier_transform,
!             get_ld_rcut, get_logarithmic_derivative, get_output_pp,
!             get_pp_output, get_pseudopotential, get_debug, read_std_form,
!             write_test, write_std_form, read_gncpp_form, write_gncpp_form,
!             psv_skip_comment, get_psv_allel, get_psv_psv, get_psv_solps,
!             get_psv_drv, read_psv_form, get_init_energy, write_psv_form,
!             getfiletype_name, main
!
!  AUTHOR: K. Mae   August, 2003
!  
!  The license of the code and contact address:
!  See the files, COPYRIGHT and LICENSE (or LICENSE_J.pdf)                  
!------------------------------------------------------------------------------

Pseudopotential input-file conversion
gncpp <-> standard <-> psv

read_std_form()
write_std_form()
read_gncpp_form()
write_gncpp_form()
read_psv_form()
write_psv_form()

last modification: 2003.8.21 by K. Mae
*********************************************************************************/
#include "ppconv.h"

/****************************************************
Tool functions
*****************************************************/
char logbuf1[128];
char logbuf2[128];
int strupper( char *str )
{
int i;

	for( i = 0; str[i] != 0; i++ ){
		str[i] = toupper( str[i] );
	}
	return( i );
}

char constbuf[128];
char *constout( char *str )
{
int i;

	strcpy( constbuf, str );
	strupper( constbuf );
	return( constbuf );
}


int strlower( char *str )
{
int i;

	for( i = 0; str[i] != 0; i++ ){
		str[i] = tolower( str[i] );
	}
	return( i );
}

int getlistno( char *str, char **list, int nlist )
{
int i, ret;

	strlower( str );
	for( i = 0; i < nlist; i++ ){
		if( strcmp( str, list[i] ) == 0 ){
			break;
		}
	}
	ret = (i >= nlist)? -1: i;
	return( ret );
}

double Dtolf( double a, int n, int sign )
{
int i;
double ret, mul;

	ret = a;
	mul = (sign > 0)? 10.0: 0.1;
	for( i = 0; i < n; i++ ){
		ret *= mul;
	}
	
	return( ret );
}

double atod( char *buf )
{
int len, i, j1, j2, point, de, esign;
char c, c1[64], c2[16];
double ret;

	point = de = 0;
	j1 = j2 = 0;
	len = strlen( buf );
	for( i = 0; i < len; i++ ){
		c = buf[i];
		if( c == 'd' || c == 'D' ||c == 'e' ||c == 'E' ){
			de = 1;
			i++;
			esign = ( buf[i] == '-' )? -1: 1;
			continue;
		}
		else if( c == '.' ){
			point = 1;
		}
		if( de ){
			c2[j2] = c;
			j2++;
		}
		else{
			c1[j1] = c;
			j1++;
		}
	}
	if( point ){
		c1[j1] = '0';
		j1++;
	}
	c1[j1] = c2[j2] = 0;
	if( de ){
		ret = Dtolf( atof(c1), atoi(c2), esign );
	}
	else{
		ret = atof( c1 );
	}

	return( ret );
}

char *dtoa( char *buf, double a )
{
int i;

	sprintf( buf,"%e", a );
	for( i = 0; buf[i] != 0; i++ ){
		if( buf[i] == 'e' || buf[i] == 'E' || buf[i] == 'd' || buf[i] == 'D'){
			buf[i] = 'e';
		}
	}

	return( buf );
}

int gotonextline( char *buf, int p, int imax )
{
int i;

	for( i = p; i < imax; i++ ){
		if( buf[i] == 0x0a || buf[i] == 0x1a ){
			i++;
			break;
		}
	}
	return( i );
}

int skip_comment( char *buf, int p, int imax )
{
int i;

	i = p+2;
	while( i < imax-1 ){
		if( buf[i] == '*' && buf[i+1] == '/' ){
			i += 2;
			break;
		}
		i++;
	}
	if( i >= imax-1 ){
		i = imax;
	}
	return( i );
}



int skip_space( char *buf, int p, int imax )
{
int i;

	i = p;
	while( i < imax ){
		while( i < imax-1 ){
			if( buf[i] == '/' && buf[i+1] == '*' ){
				if( (i = skip_comment( buf, i, imax )) == imax ){
					return( imax );
				}
			}
			else{
				break;
			}
		}
		if( buf[i] != ' ' && buf[i] != 0x0d && buf[i] != 0x0a /*&& buf[i] != TAB*/){
			break;
		}
		i++;
	}
	return( i );
}

int skip_k2k( char *buf, int p, int imax )
{
int i, kakko = 0;

	if( (i = skip_space( buf, p, imax )) == imax ){
		return( -999 );
	}
	if( buf[i] != '{' ){
		return( i );
	}
	kakko++;
	i++;
	while( i < imax ){
		if( buf[i] == '}' ){
			kakko--;
			if( kakko == 0 ){
				i++;
				break;
			}
		}
		else if( buf[i] == '{' ){
			kakko++;
		}
		else if( buf[i] == '/' && buf[i+1] == '*' ){
			if( (i = skip_comment( buf, i, imax )) == imax ){
				break;
			}
			continue;
		}
		i++;
	}
	if( i >= imax ){
		i = imax;
	}
	return( i );
}

int get_keyword( char *key, char *buf, int *p, int imax )
{
int i, j, n;
char c;

	j = 0;
	if( (i = skip_space( buf, *p, imax )) == imax ){
		*p = i;
		return( -999 );
	}
	while( i < imax ){
		c = buf[i];
		if( c == ' ' || c == '{' || (c == '/' && buf[i+1] == '*') || c == 0x1A ){
			break;
		}
		key[j] = c;
		i++;
		j++;
	}

	key[j] = 0;
	*p = i;
	if( key[0] == '}' ){
		return(999);
	}
	if( (n = getlistno( key, keylist, NKEYLIST )) < 0 ){
		printf("get_keyword: illegal keyword found\n");
		fprintf(fpl,"get_keyword: illegal keyword found\n");
		return ( -1 );
	}
	return( keyno[n] );
}

int get_parastr( char *pstr, char *buf, int *p, int imax )
{
int i, j;
char c;

	j = 0;
	if( (i = skip_space( buf, *p, imax )) == imax ){
		*p = i;
		return( -999 );
	}
	while( i < imax ){
		c = buf[i];
		if( c == ' ' || c == ',' || c == '}' || (c == '/' && buf[i+1] == '*') || c == 0x0a || c == 0x1a ){
			break;
		}
		pstr[j] = c;
		i++;
		j++;
	}
	pstr[j] = 0;
	*p = i;

	return( 0 );
}


int get_kakko_l( char *buf, int *p, int imax )
{
int i, ret = 0;

	if( (i = skip_space( buf, *p, imax )) == imax ){
		*p = i;
		return( -999 );
	}
	if( buf[i] == '}' ){
		ret = 999;
	}
	else if( buf[i] != '{' ){
		printf("left parenthesis not found\n");
		fprintf(fpl,"left parenthesis not found\n");
		ret = -1;
	}
	i++;
	*p = i;
	return( ret );
}

int get_kakko_r( char *buf, int *p, int imax )
{
int i, ret = 0;

	if( (i = skip_space( buf, *p, imax )) == imax ){
		*p = i;
		return( -999 );
	}
	if( buf[i] != '}' ){
		printf("right parenthesis not found\n");
		fprintf(fpl,"right parenthesis not found\n");
		ret = -1;
	}
	i++;
	*p = i;
	return( ret );
}

int get_komma( char *buf, int *p, int imax )
{
int i, ret = 0;

	if( (i = skip_space( buf, *p, imax )) == imax ){
		*p = i;
		return( -999 );
	}

	if( buf[i] == '}' ){
		ret = KN_END;
	}
	else if( buf[i] != ',' ){
		printf("comma not found\n");
		fprintf(fpl,"comma not found\n");
		ret = -1;
	}
	i++;
	*p = i;
	return( ret );
}

int get_mesh( int *ma, int *val0, double *val1, double *val2 )
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = *ma = getlistno( pbuf, m_or_a, NMORA );
	if( ret < 0 ){
		printf("mesh: input MANUAL or AUTOMATIC\n");
		fprintf(fpl,"mesh: input MANUAL or AUTOMATIC\n");
		return( ret );
	}
	get_kakko_r( rbuf, &rptr, rlen );

	if( *ma == 0 ){
		get_kakko_l( rbuf, &rptr, rlen );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		*val0 = atoi( pbuf );
		get_komma( rbuf, &rptr, rlen );

		get_parastr( pbuf, rbuf, &rptr, rlen );
		*val1 = atof( pbuf );
		get_komma( rbuf, &rptr, rlen );

		get_parastr( pbuf, rbuf, &rptr, rlen );
		*val2 = atof( pbuf );
		get_kakko_r( rbuf, &rptr, rlen );
	}
	else{
		rptr = skip_k2k( rbuf, rptr, rlen );
		*val0 = 0;
		*val1 = *val2 = 0.0;
	}

	return( ret );
}
/**************************************************/

/****************************************************
standars format reading functions
****************************************************/
int get_title()
{
int ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( title, rbuf, &rptr, rlen );
	get_kakko_r( rbuf, &rptr, rlen ); 

	/*printf("title: normal\n");*/
	fprintf(fpl,"title: normal\n");

	return( ret );
}

int get_spin()
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = spin = getlistno( pbuf, spin_alt, NSPINALT );
	if( ret < 0 ){
		printf("spin: input RESTRICTED or POLARIZED\n");
		fprintf(fpl,"spin: input RESTRICTED or POLARIZED\n");
		return( ret );
	}
	get_komma( rbuf, &rptr, rlen ); 
	get_parastr( pbuf, rbuf, &rptr, rlen );
	spin_val = atof( pbuf );
	get_kakko_r( rbuf, &rptr, rlen ); 

	/*printf("spin: normal\n");*/
	fprintf(fpl,"spin: normal\n");
	return( ret );
}

int get_ks_equation()
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = ks_equation = getlistno( pbuf, ks_equation_alt, NKSEQUATIONALT );
	if( ret < 0 ){
		printf("ks_equation: input NONREL, SREL, REL, or SREL_GNCPP\n");
		fprintf(fpl,"ks_equation: input NONREL, SREL, REL, or SREL_GNCPP\n");
		return( ret );
	}
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("ks_equation: normal\n");*/
	fprintf(fpl,"ks_equation: normal\n");
	return( ret );
}

int get_configuration()
{
int i, ret = 0, rt, downfg = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	num_cnf = 0;
	num_occ = 0;
	while( 1 ){
		ret = get_kakko_l( rbuf, &rptr, rlen );
		if( ret == 999 || ret < 0){
			break;
		}
		get_parastr( pbuf, rbuf, &rptr, rlen );
		i = getlistno( pbuf, orbital_alt, NORBITALALT );
		if( i < 0 ){
			printf("configuration: strange orbital name found '%s'\n", pbuf);
			fprintf(fpl,"configuration: input strange orbital name found '%s'\n",pbuf);
			return( -1 );
		}
		conf[num_cnf].orbital = i;
		rt = get_komma( rbuf, &rptr, rlen );

		get_parastr( pbuf, rbuf, &rptr, rlen );
		conf[num_cnf].up = atof( pbuf );
		if( conf[num_cnf].up > 0.0 ){
			num_occ++;
		}
		rt = get_komma( rbuf, &rptr, rlen );
		if( rt == KN_END ){
			num_cnf++;
			downfg = 0;
			continue;
		}
		get_parastr( pbuf, rbuf, &rptr, rlen );
		conf[num_cnf].down = atof( pbuf );
		get_kakko_r( rbuf, &rptr, rlen ); 
		num_cnf++;
		downfg = 1;
	}

	/**********************************
	Get algorithm from Dr. Okamoto
	*********************************/
	if( spin == 1 && downfg == 0){	/*polarized and no input for down spin*/
		;
	}
	else if( downfg == 0 ){
		for( i = 0; i < num_cnf; i++ ){
			conf[num_cnf].down = 0.0;
		}
	}

	ret = num_cnf;

	/*printf("configuration: normal\n");*/
	fprintf(fpl,"configuration: normal\n");
	return( ret );
}

int get_core_potential()
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = core_potential = getlistno( pbuf, core_potential_alt, NCOREPOTENTIALALT );
	if( ret < 0 ){
		printf("core_potential: input NORMAL or PATOM\n");
		fprintf(fpl,"core_potential: input NORMAL or PATOM\n");
		return( ret );
	}
	get_kakko_r( rbuf, &rptr, rlen ); 

	/*printf("core_potential: normal\n");*/
	fprintf(fpl,"core_potential: normal\n");
	return( ret );
}

int get_xc_potential()
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = xc_potential = getlistno( pbuf, xc_potential_alt, NXCPOTENTIALALT );
	if( ret < 0 ){
		printf("xc_potential: input LDAPZ81, LDAPW92, GGAPBE96 or GGAREVPBE\n");
		fprintf(fpl,"xc_potential: input LDAPZ81, LDAPW92, GGAPBE96 or GGAREVPBE\n");
		return( ret );
	}
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("xc_potential: normal\n");*/
	fprintf(fpl,"xc_potential: normal\n");
	return( ret );
}


int get_mix()
{
int ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	mix = atof( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("mix: normal\n");*/
	fprintf(fpl,"mix: normal\n");
	return( ret );
}

int get_anderson()
{
int ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	if( pbuf[0] == 0 ){
		anderson = -1;
	}
	else{
		anderson = atoi( pbuf );
	}
	get_kakko_r( rbuf, &rptr, rlen );

	if( anderson >= 0){
		/*printf("anderson: normal\n");*/
		fprintf(fpl,"anderson: normal\n");
	}
	return( ret );
}

int get_mixing()
{
int kn, ret = 0;

	anderson = -1;
	get_kakko_l( rbuf, &rptr, rlen );

	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_MIX:		ret = get_mix();		break;
		case KN_ANDERSON:	ret = get_anderson(); break;
		default:	printf("get_mixing: illeagal keyword found: %d !!!\n", kn);
					fprintf(fpl,"get_mixing: illeagal keyword found: %d !!!\n", kn);
					return( -1 );
		}
	}

	/*printf("mixing: normal\n");*/
	fprintf(fpl,"mixing: normal\n");
	return( ret );
}


int get_energy()
{
int ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	conv_scf_e = atof( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("energy: normal\n");*/
	fprintf(fpl,"energy: normal\n");
	return( ret );
}

int get_potential()
{
int ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	conv_scf_p = atof( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("potential: normal\n");*/
	fprintf(fpl,"potential: normal\n");
	return( ret );
}

int get_max_iteration()
{
int ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	conv_max_iteration = atoi( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("max_iteration: normal\n");*/
	fprintf(fpl,"max_iteration: normal\n");
	return( ret );
}


int get_conv_scf()
{
int kn, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );

	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_ENERGY:			ret = get_energy();			break;
		case KN_POTENTIAL:		ret = get_potential();		break;
		case KN_MAX_ITERATION:	ret = get_max_iteration();	break;
		default:	printf("get_conv_scf: illeagal keyword found: %d !!!\n", kn);
					fprintf(fpl,"get_conv_scf: illeagal keyword found: %d !!!\n", kn);
					return( -1 );
		}
	}

	/*printf("scf: normal\n");*/
	fprintf(fpl,"scf: normal\n");
	return( ret );
}

int get_conv_wf()
{
int ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	conv_wf = atof( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("wf: normal\n");*/
	fprintf(fpl,"wf: normal\n");
	return( ret );
}

int get_convergence()
{
int kn, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );

	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_CONV_SCF:	ret = get_conv_scf();	break;
		case KN_WF:			ret = get_conv_wf();	break;
		default:	printf("get_convergence: illeagal keyword found: %d !!!\n", kn);
					fprintf(fpl,"get_convergence: illeagal keyword found: %d !!!\n", kn);
					return( -1 );
		}
	}

	/*printf("convergence: normal\n");*/
	fprintf(fpl,"convergence: normal\n");
	return( ret );
}


int get_predictor_corrector()
{
int ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	pre_cor0 = atoi( pbuf );
	get_komma( rbuf, &rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	pre_cor1 = atoi( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("predictor_corrector: normal\n");*/
	fprintf(fpl,"predictor_corrector: normal\n");
	return( ret );
}

int get_diff_order()
{
int ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	diff_order = atoi( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("diff_order: normal\n");*/
	fprintf(fpl,"diff_order: normal\n");
	return( ret );
}

int get_num_coef_veff()
{
int ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	num_coeff_veff = atoi( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("num_coef_veff: normal\n");*/
	fprintf(fpl,"num_coef_veff: normal\n");
	return( ret );
}

int get_output_para()
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	i = getlistno( pbuf, output_alt, NOUTPUTALT );
	if( i < 0 ){
		printf("output: input NONE, IGOR, or GNUPLOT\n");
		fprintf(fpl,"output: input NONE, IGOR, or GNUPLOT\n");
		return( -1 );
	}
	else{
		ret = i;
	}
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("num_coef_veff: normal\n");*/
	fprintf(fpl,"num_coef_veff: normal\n");
	return( ret );
}


int get_ae_output()
{
int kn, ret = 0;

	output_rho = 0;
	output_rm = 0;
	get_kakko_l( rbuf, &rptr, rlen );

	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_OUTPUT_RHO: ret = output_rho = get_output_para();	break;
		case KN_OUTPUT_RM:	ret = output_rm = get_output_para();	break;
		default:	printf("get_ae_output: illeagal keyword found: %d !!!\n", kn);
					fprintf(fpl,"get_ae_output: illeagal keyword found: %d !!!\n", kn);
					return( -1 );
		}
	}

	/*printf("output: normal\n");*/
	fprintf(fpl,"output: normal\n");
	return( ret );
}




int get_all_electron()
{
int kn, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );

	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_TITLE:			ret = get_title();			break;
		case KN_SPIN:			ret = get_spin();			break;
		case KN_KS_EQUATION:	ret = get_ks_equation();	break;
		case KN_CONFIGURATION:	ret = get_configuration();	break;
		case KN_CORE_POTENTIAL:	ret = get_core_potential(); break;
		case KN_XC_POTENTIAL:	ret = get_xc_potential();	break;
		case KN_RMESH:			ret = get_mesh( &rmesh, &rmesh_val0, &rmesh_val1, &rmesh_val2); break;
		case KN_MIXING:			ret = get_mixing();			break;
		case KN_CONVERGENCE:	ret = get_convergence();	break;
		case KN_PREDICTOR_CORRECTOR: ret = get_predictor_corrector(); break;
		case KN_DIFF_ORDER:		ret = get_diff_order();		break;
		case KN_NUM_COEFF_VEFF: ret = get_num_coef_veff();	break;
		case KN_OUTPUT:			ret = get_ae_output();		break;
		default:	printf("illeagal keyword found: %d !!!\n", kn);
					fprintf(fpl,"illeagal keyword found: %d !!!\n", kn);
					return( -1 );
		}
	}

	/*printf("all_electron: normal\n");*/
	fprintf(fpl,"all_electron: normal\n");
	return( ret );
}
/****************************************************/
int get_shape( int *sh, int *val )
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = *sh = getlistno( pbuf, shape_alt, NSHAPEALT );
	if( ret < 0 ){
		printf("shape: input POLYNOMIAL, SBESSEL, TM90, or KERKER85\n");
		fprintf(fpl,"shape: input POLYNOMIAL, SBESSEL, TM90, or KERKER85\n");
		return( ret );
	}
	get_kakko_r( rbuf, &rptr, rlen );

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	*val = atoi( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("shape: normal\n");*/
	fprintf(fpl,"shape: normal\n");
	return( ret );
}

int get_spdf( int spdf )
{
int i, ret = 0, kn;
struct WF_SPDF *w;


	if( spdf > 3 ){
		printf("spdf: spdf should 0-3\n");
		fprintf(fpl,"spdf: spdf should 0-3\n");
		return( -1 );
	}
	w = &wf_l[spdf];
	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = w->orbital = getlistno( pbuf, orbital_alt, NORBITALALT );
	if( ret < 0 ){
		printf("spdf: strange orbital name found\n");
		fprintf(fpl,"spdf: strange orbital name found\n");
		return( -1 );
	}

	get_kakko_r( rbuf, &rptr, rlen );

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = w->pptype = getlistno( pbuf, pptype_alt, NPPTYPEALT );
	if( ret < 0 ){
		printf("spdf: input NC or US\n");
		fprintf(fpl,"spdf: input NC or US\n");
		return( -1 );
	}

	get_kakko_r( rbuf, &rptr, rlen );

	get_kakko_l( rbuf, &rptr, rlen );
	w->num_rcut = 0;
	while( 1 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		else if( kn == KN_RCUT ){
			get_kakko_l( rbuf, &rptr, rlen );
			get_parastr( pbuf, rbuf, &rptr, rlen );
			w->rcut[w->num_rcut][0] = atof( pbuf );
			get_komma( rbuf, &rptr, rlen );

			get_parastr( pbuf, rbuf, &rptr, rlen );
			w->rcut[w->num_rcut][1] = atof( pbuf );
			get_kakko_r( rbuf, &rptr, rlen );
			w->num_rcut++;
		}
		else{
			printf("spdf: illeagal keyword found: %d !!!\n", kn);
			fprintf(fpl,"spdf: illeagal keyword found: %d !!!\n", kn);
			return( -1 );
		}
	}

	/*printf("spdf: normal\n");*/
	fprintf(fpl,"spdf: normal\n");
	return( ret );
}

int get_nc()
{
int kn, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );

	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_SHAPE: 	ret = get_shape( &wf_nc_shape, &wf_nc_shape_val ); break;
		default:	printf("nc: illeagal keyword found: %d !!!\n", kn);
					fprintf(fpl,"nc: illeagal keyword found: %d !!!\n", kn);
					return( -1 );
		}
	}

	/*printf("nc: normal\n");*/
	fprintf(fpl,"nc: normal\n");
	return( ret );
}

int get_us()
{
int kn, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );

	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_SHAPE: 	ret = get_shape( &wf_us_shape, &wf_us_shape_val ); break;
		default:	printf("us: illeagal keyword found: %d !!!\n", kn);
					fprintf(fpl,"us: illeagal keyword found: %d !!!\n", kn);
					return( -1 );
		}
	}

	/*printf("us: normal\n");*/
	fprintf(fpl,"us: normal\n");
	return( ret );
}

int get_wf()
{
int i, kn, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );

	for( i = 0; i < 4; i++ ){
		wf_l[i].orbital = -1;
	}
	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_S:
		case KN_P:
		case KN_D:
		case KN_F:	ret = get_spdf( kn - KN_S ); break;
		case KN_NC: ret = get_nc(); break;
		case KN_US: ret = get_us(); break;
		case KN_GMESH:	get_mesh( &wf_gmesh, &wf_gmesh_val0, &wf_gmesh_val1, &wf_gmesh_val2 ); break;
		default:		printf("wf: illeagal keyword found: %d !!!\n", kn);
						fprintf(fpl,"wf: illeagal keyword found: %d !!!\n", kn);
						return( -1 );
		}
	}
	/*printf("wf: normal\n");*/
	fprintf(fpl,"wf: normal\n");
	return( ret );
}

int get_local()
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = local = getlistno( pbuf, local_alt, NLOCALALT );
	if( ret < 0 ){
		printf("local: input ORBITAL, GNCPP or PSV\n");
		fprintf(fpl,"local: input ORBITAL, GNCPP or PSV\n");
		return( -1 );
	}
	get_kakko_r( rbuf, &rptr, rlen );

	if( local == 0 || local == 2 ){	/*orbital or psv*/
		get_kakko_l( rbuf, &rptr, rlen );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		ret = local_val = getlistno( pbuf, spdf_alt, NSPDFALT );
		if( ret < 0 ){
			printf("local: input S,P,D, or F\n");
			fprintf(fpl,"local: input S,P,D, or F\n");
			return( -1 );
		}
		get_kakko_r( rbuf, &rptr, rlen );
	}
	else{	/*gncpp*/
		get_kakko_l( rbuf, &rptr, rlen );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		local_rcloc = atof( pbuf );
		get_komma( rbuf, &rptr, rlen );

		get_parastr( pbuf, rbuf, &rptr, rlen );
		local_const = atof( pbuf );
		get_komma( rbuf, &rptr, rlen );

		get_parastr( pbuf, rbuf, &rptr, rlen );
		local_lamda = atof( pbuf );
		get_kakko_r( rbuf, &rptr, rlen );
	}

	/*printf("local: normal\n");*/
	fprintf(fpl,"local: normal\n");
	return( ret );
}

int get_qps_rcut( int n )
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );

	for( i = 0; i < 5; i++ ){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		qps_rcut[n][i] = atoi( pbuf );
		if( i == 4 ){
			get_kakko_r( rbuf, &rptr, rlen );
		}
		else{
			get_komma( rbuf, &rptr, rlen );
		}
	}

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	qps_rcut_val[n] = atof( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("qps_rcut: normal\n");*/
	fprintf(fpl,"qps_rcut: normal\n");
	return( ret );
}

int get_qps()
{
int kn, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );

	qps_num_rcut = 0;
	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_RCUT:	ret = get_qps_rcut( qps_num_rcut ); 
						qps_num_rcut++;
						break;
		case KN_SHAPE:	ret = get_shape( &qps_shape, &qps_shape_val ); break;
		case KN_GMESH:	ret = get_mesh( &qps_gmesh, &qps_gmesh_val0, &qps_gmesh_val1, &qps_gmesh_val2); break;
		default:		printf("qps: illeagal keyword found: %d !!!\n", kn);
						fprintf(fpl,"qps: illeagal keyword found: %d !!!\n", kn);
						return( -1 );
		}
	}

	/*printf("qps: normal\n");*/
	fprintf(fpl,"qps: normal\n");
	return( ret );
}

int get_cc_rcut()
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = cc_rcut = getlistno( pbuf, cc_rcut_alt, NCCRCUTALT );
	if( ret < 0 ){
		printf("core_correction_rcut: input AUTOMATIC, RADIUS or CV_RATIO\n");
		fprintf(fpl,"core_correction_rcut: input AUTOMATIC, RADIUS or CV_RATIO\n");
		return( -1 );
	}
	get_kakko_r( rbuf, &rptr, rlen );

	if( cc_rcut > 0 ){
		get_kakko_l( rbuf, &rptr, rlen );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		cc_rcut_val = atof( pbuf );
		get_kakko_r( rbuf, &rptr, rlen );
	}
	else{
		cc_rcut_val = 0.0;
	}

	/*printf("core_correction_rcut: normal\n");*/
	fprintf(fpl,"core_correction_rcut: normal\n");
	return( ret );
}

int get_core_corre()
{
int kn, i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = core_corre = getlistno( pbuf, core_corre_alt, NCORECORREALT );
	if( ret < 0 ){
		printf("core_correction: input NONE, PCC or FCC\n");
		fprintf(fpl,"core_correction: input NONE, PCC or FCC\n");
		return( -1 );
	}
	get_kakko_r( rbuf, &rptr, rlen );

	if( !core_corre ){
		if( (ret = skip_k2k( rbuf, rptr, rlen )) >= 0 ){
			rptr = ret;
			ret = 0;
		}
		return( ret );
	}

	get_kakko_l( rbuf, &rptr, rlen );
	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_RCUT:	ret = get_cc_rcut(); break;
		case KN_SHAPE:	ret = get_shape( &cc_shape, &cc_shape_val ); break;
		case KN_GMESH:	ret = get_mesh( &cc_gmesh, &cc_gmesh_val0, &cc_gmesh_val1, &cc_gmesh_val2); break;
		default:		printf("core_correction: illeagal keyword found: %d !!!\n", kn);
						fprintf(fpl,"core_correction: illeagal keyword found: %d !!!\n", kn);
						return( -1 );
		}
	}
	/*printf("core_correction: normal\n");*/
	fprintf(fpl,"core_correction: normal\n");
	return( ret );
}

int get_fourier_transform()
{
int kn, i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = i = getlistno( pbuf, yes_no, NYESNO );
	if( ret < 0 ){
		is_calc_ft = -1;
		printf("fourier_transform: input YES/Y or NO/N\n");
		fprintf(fpl,"fourier_transform: input YES/Y or NO/N\n");
		return( -1 );
	}
	is_calc_ft = i/(NYESNO/2);
	get_kakko_r( rbuf, &rptr, rlen );

	if( is_calc_ft == 0 ){
		if( (ret = skip_k2k( rbuf, rptr, rlen )) >= 0 ){
			rptr = ret;
			ret = 0;
		}
		return( ret );
	}

	get_kakko_l( rbuf, &rptr, rlen );
	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_GMESH:	ret = get_mesh( &ft_gmesh, &ft_gmesh_val0, &ft_gmesh_val1, &ft_gmesh_val2 ); break;
		default:		printf("fourier_transform: illeagal keyword found: %d !!!\n", kn);
						fprintf(fpl,"fourier_transform: illeagal keyword found: %d !!!\n", kn);			
						return( -1 );
		}
	}

	/*printf("fourier_transform: normal\n");*/
	fprintf(fpl,"fourier_transform: normal\n");
	return( ret );
}

int get_ld_rcut()
{
	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ld_rcut = atof( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("ld_rcut: normal\n");*/
	fprintf(fpl,"ld_rcut: normal\n");
	return( 0 );
}

int get_logarithmic_derivative()
{
int kn, i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = i = getlistno( pbuf, yes_no, NYESNO );
	if( ret < 0 ){
		is_calc_ld = -1;
		printf("logarithmic_derivative: input YES/Y or NO/N\n");
		fprintf(fpl,"logarithmic_derivative: input YES/Y or NO/N\n");
		return( -1 );
	}
	is_calc_ld = i/(NYESNO/2);
	get_kakko_r( rbuf, &rptr, rlen );

	if( is_calc_ld == 0 ){
		if( (ret = skip_k2k( rbuf, rptr, rlen )) >= 0 ){
			rptr = ret;
			ret = 0;
		}
		return( ret );
	}

	get_kakko_l( rbuf, &rptr, rlen );
	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_EMESH:	ret = get_mesh( &ld_emesh, &ld_emesh_val0, &ld_emesh_val1, &ld_emesh_val2 ); break;
		case KN_RCUT:	ret = get_ld_rcut(); break;
		default:		printf("logarithmic_derivative: illeagal keyword found: %d !!!\n", kn);
						fprintf(fpl,"logarithmic_derivative: illeagal keyword found: %d !!!\n", kn);
						return( -1 );
		}
	}

	/*printf("logarithmic_derivative: normal\n");*/
	fprintf(fpl,"logarithmic_derivative: normal\n");
	return( ret );
}

int get_output_pp()
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );

	num_output_pp = 0;
	while(1){
		if( (ret = get_parastr( pbuf, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		ret = output_pp[num_output_pp] = getlistno( pbuf, output_pp_alt, NOUTPUTPPALT );
		if( ret < 0 ){
			printf("output_pp: illeagal parameter\n");
			fprintf(fpl,"output_pp: illeagal parameter\n");
			return( -1 );
		}
		num_output_pp++;
		if( (ret = get_komma( rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
	}

	/*printf("output_pp: normal\n");*/
	fprintf(fpl,"output_pp: normal\n");
	return( ret );
}

int get_pp_output()
{
int kn, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );

	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_OUTPUT_PP:	ret = get_output_pp(); break;
		case KN_OUTPUT_FT:	output_ft = ret = get_output_para(); break;
		case KN_OUTPUT_LOGDERI:		output_logderi = ret = get_output_para(); break;
		default:	printf("pp_output: illeagal keyword found: %d !!!\n", kn);
					fprintf(fpl,"pp_output: illeagal keyword found: %d !!!\n", kn);
					return( -1 );
		}
	}

	/*printf("pp_output: normal\n");*/
	fprintf(fpl,"pp_output: normal\n");
	return( ret );
}


int get_pseudopotential()
{
int kn, i, ret = 0;


	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = i = getlistno( pbuf, yes_no, NYESNO );
	if( ret < 0 ){
		is_calc_pp = -1;
		printf("pseudopotential: input YES/Y or NO/N\n");
		fprintf(fpl,"pseudopotential: input YES/Y or NO/N\n");
		return( -1 );
	}
	is_calc_pp = i/(NYESNO/2);
	get_kakko_r( rbuf, &rptr, rlen );

	if( is_calc_pp == 0 ){
		if( (ret = skip_k2k( rbuf, rptr, rlen )) >= 0 ){
			rptr = ret;
			ret = 0;
		}
		return( ret );
	}


	get_kakko_l( rbuf, &rptr, rlen );
	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_WF:						ret = get_wf(); break;
		case KN_LOCAL:					ret = get_local(); break;
		case KN_QPS:					ret = get_qps(); break;
		case KN_CORE_CORRE:				ret = get_core_corre(); break;
		case KN_FOURIER_TRANSFORM:		ret = get_fourier_transform(); break;
		case KN_LOGARITHMIC_DERIVATIVE:	ret = get_logarithmic_derivative(); break;
		case KN_OUTPUT:					ret = get_pp_output(); break;
		default:						printf("pseudopotential: illeagal keyword found: %d !!!\n", kn);
										fprintf(fpl,"pseudopotential: illeagal keyword found: %d !!!\n", kn);
										return( -1 );
		}
	}

	/*printf("pseudopotential: normal\n");*/
	fprintf(fpl,"pseudopotential: normal\n");
	return( ret );
}
/****************************************************/

int get_debug()
{
int i, ret = 0;

	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	ret = i = getlistno( pbuf, yes_no, NYESNO );
	if( ret < 0 ){
		is_write_debug = -1;
		printf("pseudopotential: input YES/Y or NO/N\n");
		fprintf(fpl,"pseudopotential: input YES/Y or NO/N\n");
		return( -1 );
	}
	ret = is_write_debug = i/(NYESNO/2);
	get_kakko_r( rbuf, &rptr, rlen );

	/*printf("debug: normal\n");*/
	fprintf(fpl,"debug: normal\n");
	return( ret );
}
/****************************************************/



int read_std_form( char *fname )
{
int     i, kn, ret = 0;
int     ddd;

	fprintf(fpl,"READING{STANDARD}{\n");

	if( (fpr = fopen( fname, "r" )) == NULL ){
		printf("read_std_form: read-file open error [%s]\n", fname);
		fprintf(fpl,"read_std_form: read-file open error [%s]\n", fname);
		fprintf(fpl,"}\n");
		return( -1 );
	}

	rptr = 0;
	fseek( fpr, 0, SEEK_SET );
	rlen = fread( rbuf, 1, RBUFSIZE, fpr );
	fclose( fpr );
	if( rlen < 0 ){
		printf("read_std_form: fread error: rlen=%d\n", rlen );
		fprintf(fpl,"read_std_form: fread error: rlen=%d\n", rlen );
		fprintf(fpl,"}\n");
		return( rlen );
	}

	/****************************************
	initialize variables
	****************************************/
	anderson = -1;
	pre_cor0 = 5;
	pre_cor1 = 3;
	diff_order = 4;
	num_coeff_veff =5;
	output_rho = 0;
	output_rm = 0;
	is_calc_pp = 0;
	for( i = 0; i < 4; i++ ){
		wf_l[i].orbital = -1;
		wf_l[i].pptype = 0;
	}
	wf_nc_shape = wf_us_shape = -1;
	core_corre = 0;
	is_calc_ft = 0;
	is_calc_ld = 0;
	num_output_pp = 0;
	output_ft = 0;
	output_logderi = 0;
	is_write_debug = 0;


	kn = get_keyword( keyword, rbuf, &rptr, rlen );
	if( kn != KN_ATOM ){
		printf("read_std_form: keyword 'ATOM' not found\n");
		fprintf(fpl,"read_std_form: keyword 'ATOM' not found\n");
		fprintf(fpl,"}\n");
		return( -1 );
	}
	get_kakko_l( rbuf, &rptr, rlen );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	atom = atoi( pbuf );
	get_kakko_r( rbuf, &rptr, rlen );
	get_kakko_l( rbuf, &rptr, rlen );

	while( ret >= 0 ){
		if( rptr >= rlen ){
			break;
		}
		if( (kn = get_keyword( keyword, rbuf, &rptr, rlen )) == KN_END ){
			break;
		}
		switch(kn){
		case KN_ALL_ELECTRON:		ret = get_all_electron();		break;
		case KN_PSEUDOPOTENTIAL:	ret = get_pseudopotential();	break;
		case KN_DEBUG:				ret = get_debug();				break;
		default:					printf("read_std_form: illeagal keyword found: %d\n", kn);
									fprintf(fpl,"read_std_form: illeagal keyword found: %d\n", kn);
									fprintf(fpl,"}\n", kn);
									return( -1 );
		}		
	}

	fprintf(fpl,"}\n");
	return( 0 );


}

/****************************************
test output
****************************************/
int write_test( int fg )
{
int i, j;

	return( 0 );

	if( fg ){
		if( (fpw = fopen( "test_out.txt", "w+" )) == NULL ){
			printf("Can't open write-file !!!\n");
			return( -1 );
		}
	}

	/********  write to screen   *******************/
	printf("atom=%d\n",atom);
	printf("***** all electron ***\n");
	printf("title=%s\n",title);
	printf("spin=(%d,%lf)\n",spin,spin_val);
	printf("ks_equation=%d\n", ks_equation);
	printf("configuration  num=%d\n", num_cnf);
	for( i = 0; i < num_cnf; i++ ){
		printf("(%d,%lf, %lf)\n",conf[i].orbital,conf[i].up, conf[i].down);
	}

	printf("core_potential=%d\n", core_potential);
	printf("xc_potential=%d\n", xc_potential);
	printf("rmesh=%d, (%d,%lf,%lf)\n",rmesh,rmesh_val0,rmesh_val1,rmesh_val2);
	printf("mixing=%d, (%lf,%lf)\n", mix,anderson);
	printf("scf(e=%lf,p=%lf,maxiter=%d)\n", conv_scf_e, conv_scf_p,conv_max_iteration);
	printf("wf_cnv=%lf\n", conv_wf);
	printf("output(rho=%d,rm=%d)\n",output_rho,output_rm);
	printf("pre_cor=(%d,%d)\n", pre_cor0,pre_cor1);
	printf("diff_order=%d\n", diff_order);
	printf("num_coeff_veff=%d\n", num_coeff_veff);

	printf("*** pseudopotential ***\n");
	printf("WF {\n");
	for( j = 0; j < 4; j++ ){
		printf("%s=(%d,%d",spdf_alt[j],wf_l[j].orbital,wf_l[j].pptype);
		for( i = 0; i < wf_l[j].num_rcut; i++ ){
			printf(",rc(%lf,%lf)",wf_l[j].rcut[i][0],wf_l[j].rcut[i][1]);
		}
		printf("\n");
	}

	printf("local=(%d,%d)\n", local,local_val);

	printf("qps {\n");
	for( i = 0; i < qps_num_rcut; i++ ){
		printf("rc(%d,%d,%d,%d,%d)=%lf\n",qps_rcut[i][0],qps_rcut[i][1],qps_rcut[i][2],qps_rcut[i][3],qps_rcut[i][4],qps_rcut_val[i]);
	}
	printf("shape=(%d,%d)\n",qps_shape,qps_shape_val);
	printf("gmesh=%d (%d,%lf,%lf)\n",qps_gmesh,qps_gmesh_val0,qps_gmesh_val1,qps_gmesh_val2);
	printf("}\n");

	printf("core_correction{%d} {\n",core_corre);
	printf("rc=(%d,%lf)\n",cc_rcut,cc_rcut_val);
	printf("shape=(%d,%d)\n",cc_shape,cc_shape_val);
	printf("gmesh=%d (%d,%lf,%lf)\n",cc_gmesh, cc_gmesh_val0,cc_gmesh_val1,cc_gmesh_val2);
	printf("}\n");

	printf("Fourier_Transform{%d}n",is_calc_ft);
	if(is_calc_ft){
		printf("gmesh=%d (%d,%lf,%lf)\n",ft_gmesh, ft_gmesh_val0,ft_gmesh_val1,ft_gmesh_val2);
	}
	printf("Logarithmic_Derivative{%d}\n",is_calc_ld);
	if( is_calc_ld ){
		printf("emesh=%d (%d,%lf,%lf)\n",ld_emesh, ld_emesh_val0,ld_emesh_val1,ld_emesh_val2);
	}
	printf("****** Debug *****\n");
	printf("debug=%d\n",is_write_debug);

	if( fg == 0 ){
		return( 0 );
	}
	/********  write to file   *******************/
	fprintf(fpw,"atom=%d\n",atom);
	fprintf(fpw,"***** all electron ***\n");
	fprintf(fpw,"title=%s\n",title);
	fprintf(fpw,"spin=(%d,%lf)\n",spin,spin_val);
	fprintf(fpw,"ks_equation=%d\n", ks_equation);
	fprintf(fpw,"configuration  num=%d\n", num_cnf);
	for( i = 0; i < num_cnf; i++ ){
		fprintf(fpw,"(%d,%lf, %lf)\n",conf[i].orbital,conf[i].up, conf[i].down);
	}

	fprintf(fpw,"core_potential=%d\n", core_potential);
	fprintf(fpw,"xc_potential=%d\n", xc_potential);
	fprintf(fpw,"rmesh=%d, (%d,%lf,%lf)\n",rmesh,rmesh_val0,rmesh_val1,rmesh_val2);
	fprintf(fpw,"mixing=%d, (%lf,%lf)\n", mix,anderson);
	fprintf(fpw,"scf(e=%lf,p=%lf,maxiter=%d)\n", conv_scf_e, conv_scf_p,conv_max_iteration);
	fprintf(fpw,"wf_cnv=%lf\n", conv_wf);
	fprintf(fpw,"output(rho=%d,rm=%d)\n",output_rho,output_rm);
	fprintf(fpw,"pre_cor=(%d,%d)\n", pre_cor0,pre_cor1);
	fprintf(fpw,"diff_order=%d\n", diff_order);
	fprintf(fpw,"num_coeff_veff=%d\n", num_coeff_veff);

	fprintf(fpw,"*** pseudopotential ***\n");
	fprintf(fpw,"WF {\n");
	for( j = 0; j < 4; j++ ){
		fprintf(fpw,"%s=(%d,%d",spdf_alt[j],wf_l[j].orbital,wf_l[j].pptype);
		for( i = 0; i < wf_l[j].num_rcut; i++ ){
			fprintf(fpw,",rc(%lf,%lf)",wf_l[j].rcut[i][0],wf_l[j].rcut[i][1]);
		}
		fprintf(fpw,"\n");
	}

	fprintf(fpw,"local=(%d,%d)\n", local,local_val);

	fprintf(fpw,"qps {\n");
	for( i = 0; i < qps_num_rcut; i++ ){
		fprintf(fpw,"rc(%d,%d,%d,%d,%d)=%lf\n",qps_rcut[i][0],qps_rcut[i][1],qps_rcut[i][2],qps_rcut[i][3],qps_rcut[i][4],qps_rcut_val[i]);
	}
	fprintf(fpw,"shape=(%d,%d)\n",qps_shape,qps_shape_val);
	fprintf(fpw,"gmesh=%d (%d,%lf,%lf)\n",qps_gmesh,qps_gmesh_val0,qps_gmesh_val1,qps_gmesh_val2);
	fprintf(fpw,"}\n");

	fprintf(fpw,"core_correction{%d} {\n",core_corre);
	fprintf(fpw,"rc=(%d,%lf)\n",cc_rcut,cc_rcut_val);
	fprintf(fpw,"shape=(%d,%d)\n",cc_shape,cc_shape_val);
	fprintf(fpw,"gmesh=%d (%d,%lf,%lf)\n",cc_gmesh, cc_gmesh_val0,cc_gmesh_val1,cc_gmesh_val2);
	fprintf(fpw,"}\n");

	fprintf(fpw,"Fourier_Transform{%d}n",is_calc_ft);
	if(is_calc_ft){
		fprintf(fpw,"gmesh=%d (%d,%lf,%lf)\n",ft_gmesh, ft_gmesh_val0,ft_gmesh_val1,ft_gmesh_val2);
	}
	fprintf(fpw,"Logarithmic_Derivative{%d}\n",is_calc_ld);
	if( is_calc_ld ){
		fprintf(fpw,"emesh=%d (%d,%lf,%lf)\n",ld_emesh, ld_emesh_val0,ld_emesh_val1,ld_emesh_val2);
	}
	fprintf(fpw,"****** Debug *****\n");
	fprintf(fpw,"debug=%d\n",is_write_debug);

	fclose(fpw);

	return( 0 );
}

/***********************************************************************/

/*********************************************
writing standard format
*********************************************/
int write_std_form( char *fname )
{
int i, j;
char ebuf[64], pbuf[64], wfbuf[64];

	fprintf(fpl,"WRITING{STANDARD}{\n");

	/*sprintf(wfile,"stdinp%d.txt", atom);*/
	/*printf("write-file name =");*/
	/*scanf("%s",wfile);*/
	if( (fpw = fopen( fname, "w+" )) == NULL ){
		printf("Can't open write-file [%s] !!!\n",fname);
		return( -1 );
	}

	/********  write to screen   *******************/
	write_test(0);

	/********  write to file   *******************/
	fprintf(fpw,"%s{%d}{\n", keylist[0],atom);	/*atom*/
	fprintf(fpw,"    %s{\n",keylist[1]);	/*all_electron*/
	fprintf(fpw,"        %s{atom_%d_%s_%s}\n",keylist[2],atom,xc_potential_alt[xc_potential],ks_equation_alt[ks_equation]);	/*title*/
	fprintf(fpw,"        %s{%s, %lf}\n",keylist[3], spin_alt[spin], spin_val);	/*spin*/
	fprintf(fpw,"        %s{%s}\n",keylist[4],ks_equation_alt[ks_equation]);	/*ks_equation*/
	fprintf(fpw,"        %s{\n",keylist[5]);	/*configuration*/
	for( i = 0; i < num_cnf; i++ ){
		fprintf(fpw,"            {%s, %lf, %lf}\n", orbital_alt[conf[i].orbital], conf[i].up, conf[i].down);
	}
	fprintf(fpw,"        }\n");	
	fprintf(fpw,"        %s{%s}\n",keylist[6],core_potential_alt[core_potential]);	/*core_potential*/
	fprintf(fpw,"        %s{%s}\n",keylist[7],xc_potential_alt[xc_potential]);	/*xc_potential*/
	fprintf(fpw,"        %s{%s}{%d, %s, %s}\n",keylist[8],m_or_a[rmesh],rmesh_val0,dtoa(ebuf,rmesh_val1),dtoa(pbuf,rmesh_val2));	/*rmesh*/
	fprintf(fpw,"        %s{%s{%lf}",keylist[9],keylist[10],mix);	/*mix*/
	if( anderson > 0 ){
		fprintf(fpw," anderson{%d}", anderson);	/*anderson*/
	}
	fprintf(fpw,"}\n");
	fprintf(fpw,"        convergence{\n");	/*convergence*/
	dtoa( ebuf, conv_scf_e );
	dtoa( pbuf, conv_scf_p );
	dtoa( wfbuf, conv_wf );
	fprintf(fpw,"            %s{%s{%s} %s{%s} %s{%d}}\n",keylist[13],keylist[14],ebuf,keylist[15],pbuf,keylist[16],conv_max_iteration);	/*scf*/
	fprintf(fpw,"            wf{%s}\n",wfbuf);	/*wf*/
	fprintf(fpw,"        }\n");	
	fprintf(fpw,"        output{\n");	/*output*/
	fprintf(fpw,"            %s{%s}\n",keylist[17],output_alt[output_rho]);	/*rho*/
	fprintf(fpw,"            %s{%s}\n",keylist[18],output_alt[output_rm]);	/*rm*/
	fprintf(fpw,"        }\n");	
	fprintf(fpw,"        %s{%d, %d}\n",keylist[19], pre_cor0, pre_cor1);	/*predictor_corrector*/
	fprintf(fpw,"        %s{%d}\n",keylist[20], diff_order);	/*diff_order*/
	fprintf(fpw,"        %s{%d}\n",keylist[21], num_coeff_veff);	/*num_coeff_veff*/
	fprintf(fpw,"    }\n");	
	fprintf(fpw,"    %s{%s}",keylist[22],yes_no[is_calc_pp*(NYESNO/2)]);	/*pseudopotential*/
	if( is_calc_pp ){
		fprintf(fpw,"{\n");
	}
	else{
		fprintf(fpw,"\n");
		printf("no pseudopotential calculation.\n");
		fprintf(fpl,"no pseudopotential calculation.\n");
		goto WRITEDEBUG;
	}
	fprintf(fpw,"        wf{\n");	/*wf*/
	for( i = 0; i < 4; i++ ){
		if( wf_l[i].orbital < 0 ){
			continue;
		}
		fprintf(fpw,"            %s{%s}{%s}{",keylist[27+i],orbital_alt[wf_l[i].orbital],pptype_alt[wf_l[i].pptype]);
		for( j = 0; j < wf_l[i].num_rcut; j++ ){
			if( j ){
				fprintf(fpw," ");
			}
			fprintf(fpw,"rcut{%lf, %lf}",wf_l[i].rcut[j][0],wf_l[i].rcut[j][1]);
		}
		fprintf(fpw,"}\n");
	}
	if( wf_nc_shape >= 0 ){
		fprintf(fpw,"            nc{shape{%s}{%d}}\n",shape_alt[wf_nc_shape],wf_nc_shape_val);
	}
	if( wf_us_shape >= 0 ){
		fprintf(fpw,"            us{shape{%s}{%d}}\n",shape_alt[wf_us_shape],wf_us_shape_val);
	}
	fprintf(fpw,"            gmesh{%s}{%d, %lf, %lf}\n",m_or_a[wf_gmesh],wf_gmesh_val0,wf_gmesh_val1,wf_gmesh_val2);
	fprintf(fpw,"        }\n");
	if( local == 0 ){		/* orbital */
		fprintf(fpw,"        local{%s}{%s}\n",local_alt[local],spdf_alt[local_val]);
	}
	else if( local == 1 ){	/* gncpp */
		fprintf(fpw,"        local{%s}{%lf, %lf, %lf}\n",local_alt[local],local_rcloc,local_const,local_lamda);
	}
	else{					/* psv */
		fprintf(fpw,"        local{%s}{%s}\n",local_alt[local],spdf_alt[local_val]);
	}
	fprintf(fpw,"        qps{\n");
	for( i = 0; i < qps_num_rcut; i++ ){
		fprintf(fpw,"            rcut{%d, %d, %d, %d, %d}{%lf}\n",qps_rcut[i][0],qps_rcut[i][1],qps_rcut[i][2],qps_rcut[i][3],qps_rcut[i][4],qps_rcut_val[i]);
	}
	fprintf(fpw,"            shape{%s}{%d}\n",shape_alt[qps_shape],qps_shape_val);
	fprintf(fpw,"            gmesh{%s}{%d, %lf, %lf}\n",m_or_a[qps_gmesh],qps_gmesh_val0,qps_gmesh_val1,qps_gmesh_val2);
	fprintf(fpw,"        }\n");

	fprintf(fpw,"        core_correction{%s}",core_corre_alt[core_corre]);
	if( core_corre ){
		fprintf(fpw,"{\n");
	}
	else{
		fprintf(fpw,"\n");
		/*printf("no core correction.\n");*/
		fprintf(fpl,"no core correction.\n");
		goto WRITEFT;
	}
	fprintf(fpw,"            rcut{%s}{%lf}\n",cc_rcut_alt[cc_rcut],cc_rcut_val);
	fprintf(fpw,"            shape{%s}{%d}\n",shape_alt[cc_shape],cc_shape_val);
	fprintf(fpw,"            gmesh{%s}{%d, %lf, %lf}\n",m_or_a[cc_gmesh],cc_gmesh_val0,cc_gmesh_val1,cc_gmesh_val2);
	fprintf(fpw,"        }\n");
			
WRITEFT:
	fprintf(fpw,"        fourier_transform{%s}",yes_no[is_calc_ft*(NYESNO/2)]);
	if( is_calc_ft ){
		fprintf(fpw,"{\n");
	}
	else{
		fprintf(fpw,"\n");
		/*printf("no fourier transform.\n");*/
		fprintf(fpl,"no fourier transform.\n");
		goto WRITELD;
	}
	fprintf(fpw,"            gmesh{%s}{%d, %lf, %lf}\n",m_or_a[ft_gmesh],ft_gmesh_val0,ft_gmesh_val1,ft_gmesh_val2);
	fprintf(fpw,"        }\n");

WRITELD:
	fprintf(fpw,"        logarithmic_derivative{%s}",yes_no[is_calc_ld*(NYESNO/2)]);
	if( is_calc_ld ){
		fprintf(fpw,"{\n");
	}
	else{
		fprintf(fpw,"\n");
		/*printf("no logarithmic_derivative.\n");*/
		fprintf(fpl,"no logarithmic_derivative.\n");
		goto WRITEDEBUG;
	}
	fprintf(fpw,"            emesh{%s}{%d, %lf, %lf}\n",m_or_a[ld_emesh],ld_emesh_val0,ld_emesh_val1,ld_emesh_val2);
	fprintf(fpw,"            rcut{%lf}\n",ld_rcut);
	fprintf(fpw,"        }\n");

	fprintf(fpw,"        output{\n");
	fprintf(fpw,"            pp{");
	for( i = 0; i < num_output_pp; i++ ){
		fprintf(fpw,"%s",output_pp_alt[output_pp[i]]);
		if( i < num_output_pp-1 ){
			fprintf(fpw,", ");
		}
	}
	fprintf(fpw,"}\n");
	fprintf(fpw,"            ft{%s}\n",output_alt[output_ft]);
	fprintf(fpw,"            logderi{%s}\n",output_alt[output_logderi]);
    fprintf(fpw,"        }\n");

	fprintf(fpw,"    }\n");

WRITEDEBUG:
	fprintf(fpw,"    debug{%s}\n",yes_no[is_write_debug*(NYESNO/2)]);

	fprintf(fpw,"}\n");
	fclose(fpw);

	fprintf(fpl,"}\n");
	return( 0 );
}



/***********************************************************************/

/********************************************
Reading gncpp format
********************************************/
int read_gncpp_form( char *fname )
{
int     ret = 0, i, j, n, l;
int     ddd, ncfg, usfg;
double nup, ndown;

	fprintf(fpl,"READING{GNCPP--->ATOM(internal)}{\n");

	if( (fpr = fopen( fname, "r" )) == NULL ){
		printf("gncpp file open error [%s]\n",fname);
		fprintf(fpl,"gncpp file open error [%s]\n",fname);
		scanf("%d",&ddd);
		return( -1 );
	}

	rptr = 0;
	fseek( fpr, 0, SEEK_SET );
	rlen = fread( rbuf, 1, RBUFSIZE, fpr );
	fclose( fpr );
	if( rlen < 0 ){
		printf("gncpp file read error\n");
		fprintf(fpl,"gncpp file read error\n");
		return( rlen );
	}

	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_mesh = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_xh = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_rmax = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_rinf = atof( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_pmix = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_edel = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_vdel = atof( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_jprt = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_jrh = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_ifp = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_nshl = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_irela = atoi( pbuf );					/*attention!! No input for non-rel*/
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_xc = getlistno( pbuf, gn_xc_alt, NGNXCALT );
	if( gn_xc < 0 ){
		printf("xc: invalid parameter\n");
		fprintf(fpl,"xc: invalid parameter\n");
		return( -1 );
	}
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_natom = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_xion = atof( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	nup = ndown = 0.0;
	for( i = 0; i < gn_nshl; i++ ){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_nljc[i] = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_wnlj[i] = atof( pbuf );
		if( gn_ifp ){
			get_parastr( pbuf, rbuf, &rptr, rlen );
			gn_down[i] = atof( pbuf );
		}
		else{
			gn_down[i] = 0.0;
		}
		nup += gn_wnlj[i];
		ndown += gn_down[i];
		rptr = gotonextline( rbuf, rptr, rlen );
	}

	get_parastr( pbuf, rbuf, &rptr, rlen );
	if( pbuf[0] >= '0' && pbuf[0] <= '9' ){
		gn_lpsmax = atoi( pbuf );
	}
	else{
		rptr = gotonextline( rbuf, rptr, rlen );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_lpsmax = atoi( pbuf );
	}
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_iloc = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_ivan = atoi( pbuf );
	if( gn_iloc == 6 ){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_rcloc = atof( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_const = atof( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_lamda = atof( pbuf );
	}
	rptr = gotonextline( rbuf, rptr, rlen );

	for( i = 0; i < gn_lpsmax; i++ ){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_wf[i].l1 = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_wf[i].itau = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_wf[i].ivanl = atoi( pbuf );
		rptr = gotonextline( rbuf, rptr, rlen );
		for( j = 0; j < gn_wf[i].itau; j++ ){
			get_parastr( pbuf, rbuf, &rptr, rlen );
			gn_wf[i].norb[j] = atoi( pbuf );
			get_parastr( pbuf, rbuf, &rptr, rlen );
			gn_wf[i].r_cl[j] = atof( pbuf );
			get_parastr( pbuf, rbuf, &rptr, rlen );
			gn_wf[i].etau[j] = atof( pbuf );
			rptr = gotonextline( rbuf, rptr, rlen );
		}
	}

	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_gmaxwf = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_meshgw = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_mopsw = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_gmaxps = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_meshgs = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_mopsc = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	i = 0;
	while( 1 ){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_il1[i] = atoi( pbuf )-1;
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_t1[i] = atoi( pbuf )-1;
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_il2[i] = atoi( pbuf )-1;
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_t2[i] = atoi( pbuf )-1;
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_il3[i] = atoi( pbuf )-1;
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_rcmlt[i] = atof( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_il4[i] = atoi( pbuf );
		rptr = gotonextline( rbuf, rptr, rlen );
		if( gn_il4[i] == 0 || rptr >= rlen ){
			break;
		}
		i++;
	}
	gn_num_qps = i+1;

	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_ipcc = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	if( gn_ipcc >= 1 ){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		gn_crc0 = atof( pbuf );
		rptr = gotonextline( rbuf, rptr, rlen );
	}
	else{
		gn_crc0 = 0.0;
	}

	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_gmaxpc = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_meshgc = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	gn_mopcc = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	/*fprintf(fpl,"}\n");*/
	/*fprintf(fpl,"CONVERSION{\n");*/

	/**** put gncpp variable to std. variable ****/
	atom = gn_natom;
	fprintf(fpl,"atom = natom\n");	/*log*/
	sprintf( title, "atom_%d", atom );
	spin = gn_ifp;
	fprintf(fpl,"spin = ifp\n");	/*log*/
	if( gn_ifp ){
		spin_val = nup-ndown;
		fprintf(fpl,"spin_val = (total number of up spin)-(total number of down spin)\n");	/*log*/
	}
	else{
		spin_val = 0.0;
		fprintf(fpl,"spin_val = 0.0 : fixed when non-polarized\n");	/*log*/
	}
	if(gn_irela == 2){
		ks_equation = 3;
	}
	else{
		ks_equation = gn_irela;
	}
	fprintf(fpl,"ks_equation = %s\n",constout(ks_equation_alt[ks_equation]));	/*log*/
	num_cnf = gn_nshl;
	fprintf(fpl,"configuration: number of orbitals=nshl\n");	/*log*/
	for( i = 0; i < num_cnf; i++ ){
		n = gn_nljc[i]/100;
		l = (gn_nljc[i] % 100)/10;
		for( j = 0; j < 19; j++ ){
			if( nl[j][0] == n && nl[j][1] == l ){
				break;
			}
		}
		conf[i].orbital = j;
		conf[i].up = gn_wnlj[i];
		conf[i].down = gn_down[i];
	}
	fprintf(fpl,"configuration: orbital from nljc/100 and (nljc%100)/10\n");	/*log*/
	fprintf(fpl,"configuration: up = wnlj\n");	/*log*/
	fprintf(fpl,"configuration: down = down\n");	/*log*/
	core_potential = 0;	/*fixed
	fprintf(fpl,"core_potential = %s : fixed\n",constout(core_potential_alt[core_potential]));	/*log*/
	xc_potential = gn_xc;
	strcpy( logbuf1, gn_xc_alt[gn_xc] );
	strcpy( logbuf2, xc_potential_alt[xc_potential] );
	strupper( logbuf1 );
	strupper( logbuf2 );
	fprintf(fpl,"xc_potential: %s--->%s\n",logbuf1,logbuf2);	/*log*/
	rmesh = 0;	/*fixed*/
	fprintf(fpl,"rmesh = %s\n",constout(m_or_a[rmesh]));	/*log*/
	rmesh_val0 = gn_mesh;
	fprintf(fpl,"rmesh_val0 = mesh\n");	/*log*/
	rmesh_val1 = gn_rmax*exp((1-gn_mesh)/gn_xh);
	fprintf(fpl,"rmesh_val1 = rmax*exp((1-mesh)/xh)\n");	/*log*/
	rmesh_val2 = gn_rmax;
	fprintf(fpl,"rmesh_val2 = rmax\n");	/*log*/
	mix = gn_pmix;
	fprintf(fpl,"mix = pmix\n");	/*log*/
	anderson = -1;	/*fixed*/
	fprintf(fpl,"anderson: not used: fixed\n");	/*log*/
	conv_max_iteration = 150;	/*fixed*/
	fprintf(fpl,"conv_max_iteration = 150 : fixed\n");	/*log*/
	conv_scf_e = gn_edel;
	fprintf(fpl,"conv_scf_e = edel\n");	/*log*/
	conv_scf_p = gn_vdel;
	fprintf(fpl,"conv_scf_p = vdel\n");	/*log*/
	conv_wf = 1.0e-10;	/*fixed*/
	fprintf(fpl,"conv_wf = 1.0e-10 : fixed\n");	/*log*/
	pre_cor0 = 5;	/*fixed*/
	fprintf(fpl,"pre_cor0 = 5 : fixed\n");	/*log*/
	pre_cor1 = 3;	/*fixed*/
	fprintf(fpl,"pre_cor1 = 3 : fixed\n");	/*log*/
	diff_order = 4;	/*fixed*/
	fprintf(fpl,"diff_order = 4 : fixed\n");	/*log*/
	num_coeff_veff = 5;	/*fixed*/
	fprintf(fpl,"num_coeff_veff = 5 : fixed\n");	/*log*/
	output_rho = 1;	/*fixed igor*/
	fprintf(fpl,"output_rho = %s : fixed\n",constout(output_alt[output_rho]));	/*log*/
	output_rm = 1;	/*fixed*/
	fprintf(fpl,"output_rm = %s : fixed\n",constout(output_alt[output_rm]));	/*log*/

	is_calc_pp = 1; /*fixed*/
	fprintf(fpl,"is_calc_pp = YES : fixed\n");	/*log*/
	ncfg = usfg = 0;
	for( i = 0; i < 4; i++ ){
		if( i >= gn_lpsmax ){
			wf_l[i].orbital = -1;
			wf_l[i].pptype = 0;
			continue;
		}
		wf_l[i].orbital = conf[gn_wf[i].norb[0]-1].orbital;
		wf_l[i].pptype = gn_wf[i].ivanl;
		if( wf_l[i].pptype == 0 ){	/*nc*/
			ncfg = 1;
		}
		else{						/*ultrasoft*/
			usfg = 1;
		}
		wf_l[i].num_rcut = gn_wf[i].itau;
		for( j = 0; j < wf_l[i].num_rcut; j++ ){
			wf_l[i].rcut[j][0] = gn_wf[i].r_cl[j];
			wf_l[i].rcut[j][1] = gn_wf[i].etau[j];
		}
	}
	fprintf(fpl,"wf: number of orbitals = lpsmax\n");	/*log*/
	fprintf(fpl,"wf: orbital from norb\n");	/*log*/
	fprintf(fpl,"wf: pptype = ivanl\n");	/*log*/
	fprintf(fpl,"wf: number of rcut = itau\n");	/*log*/
	fprintf(fpl,"wf: rcut0 = r_cl\n");	/*log*/
	fprintf(fpl,"wf: rcut1 = etau\n");	/*log*/

	wf_nc_shape = wf_us_shape = -1;
	if( ncfg ){
		wf_nc_shape = 2;	/*fixed, tm90*/
		fprintf(fpl,"wf: nc_shape = %s : fixed for norm conserving\n",constout(shape_alt[wf_nc_shape]));	/*log*/
		wf_nc_shape_val = 1;
		fprintf(fpl,"wf: nc_shape_val = 1 : fixed for norm conserving\n");	/*log*/
	}
	if( usfg ){
		wf_us_shape = 0;	/*fixed, polynomial*/
		fprintf(fpl,"wf: us_shape = %s : fixed for ultrasoft\n",constout(shape_alt[wf_us_shape]));	/*log*/
		wf_us_shape_val = gn_mopsw+1;
		fprintf(fpl,"wf: us_shape_val = mopsw+1\n");	/*log*/
	}
	wf_gmesh = 0;	/*fixed*/
	fprintf(fpl,"wf: gmesh = %s : fixed\n",constout(m_or_a[wf_gmesh]));	/*log*/
	wf_gmesh_val0 = gn_meshgw;	
	fprintf(fpl,"wf: gmesh_val0 = meshgw\n");	/*log*/
	wf_gmesh_val1 = gn_gmaxwf;	/*fixed*/
	fprintf(fpl,"wf: gmesh_val1 = gn_gmaxwf\n");	/*log*/
	wf_gmesh_val2 = 40.0;
	fprintf(fpl,"wf: gmesh_val2 = 40.0 : fixed\n");	/*log*/
	if( gn_iloc <= 4 ){
		local = 0;
		fprintf(fpl,"local = %s\n",constout(local_alt[local]));	/*log*/
		local_val = gn_iloc-1;
		fprintf(fpl,"local_val = iloc-1\n");	/*log*/
	}
	else{
		local = 1;
		fprintf(fpl,"local = %s\n",constout(local_alt[local]));	/*log*/
		local_rcloc = gn_rcloc;
		fprintf(fpl,"local_rcloc = rcloc\n");	/*log*/
		local_const = gn_const;
		fprintf(fpl,"local_const = const\n");	/*log*/
		local_lamda = gn_lamda;
		fprintf(fpl,"local_lamda = lamda\n");	/*log*/
	}
	qps_num_rcut = gn_num_qps;
	for( i = 0; i < qps_num_rcut; i++ ){
		qps_rcut[i][0] = gn_il1[i];
		qps_rcut[i][1] = gn_t1[i];
		qps_rcut[i][2] = gn_il2[i];
		qps_rcut[i][3] = gn_t2[i];
		qps_rcut[i][4] = gn_il3[i];
		qps_rcut_val[i] = gn_rcmlt[i];
	}
	fprintf(fpl,"qps: rcut0 = il1\n");	/*log*/
	fprintf(fpl,"qps: rcut1 = t1\n");	/*log*/
	fprintf(fpl,"qps: rcut2 = il2\n");	/*log*/
	fprintf(fpl,"qps: rcut3 = t2\n");	/*log*/
	fprintf(fpl,"qps: rcut4 = il3\n");	/*log*/
	fprintf(fpl,"qps: rcut_val = rcmlt\n");	/*log*/
	fprintf(fpl,"il1,t1,il2,t2,il3,rcmlt read until il4 = 0\n");	/*log*/

	qps_shape = 0;	/*fixed*/
	fprintf(fpl,"qps: shape = %s : fixed\n",constout(shape_alt[qps_shape]));	/*log*/
	qps_shape_val = gn_mopsc+1;
	fprintf(fpl,"qps: shape_val = mopsc+1\n");	/*log*/
	qps_gmesh = 0;	/*fixed*/
	fprintf(fpl,"qps: gmesh = %s : fixed\n",constout(m_or_a[qps_gmesh]));	/*log*/
	qps_gmesh_val0 = gn_meshgs;
	fprintf(fpl,"qps: gmesh_val0 = meshgs\n");	/*log*/
	qps_gmesh_val1 = gn_gmaxps;	/*fixed*/
	fprintf(fpl,"qps: gmesh_val1 = gn_gmaxps\n");	/*log*/
	qps_gmesh_val2 = 40.0;
	fprintf(fpl,"qps: gmesh_val2 = 40.0 : fixed\n");	/*log*/
	core_corre = gn_ipcc;
	fprintf(fpl,"core_correction = ipcc\n");	/*log*/
	if( gn_ipcc >= 1 ){
		if( gn_crc0 < 0.0 ){
			cc_rcut = 1;
			cc_rcut_val = -gn_crc0;
			fprintf(fpl,"core_correction: rcut = %s for crc0 < 0\n", constout(cc_rcut_alt[cc_rcut]));	/*log*/
			fprintf(fpl,"core_correction: rcut_val = -crc0 for crc0 < 0\n");	/*log*/
		}
		else{
			cc_rcut = 2;
			cc_rcut_val = gn_crc0 ;
			fprintf(fpl,"core_correction: rcut = %s for crc0 > 0\n", constout(cc_rcut_alt[cc_rcut]));	/*log*/
			fprintf(fpl,"core_correction: rcut_val = crc0 for crc0 > 0\n");	/*log*/
		}
	}
	else{ /* if no core correction*/
		cc_rcut = 1;	/*default*/
		cc_rcut_val = 2.7;	/*default*/
		fprintf(fpl,"core_correction: rcut = radius : default for no core correction\n");	/*log*/
		fprintf(fpl,"core_correction: rcut_val = 2.7 : default for no core correction\n");	/*log*/
	}
	cc_shape = 0;	/*fixed*/
	fprintf(fpl,"core_correction: shape = %s : fixed\n", constout(shape_alt[cc_shape]));	/*log*/
	cc_shape_val = gn_mopcc+1;
	fprintf(fpl,"core_correction: shape_val = mopcc+1\n");	/*log*/
	cc_gmesh = 0;	/*fixed*/
	fprintf(fpl,"core_correction: gmesh = %s ; fixed\n", constout(m_or_a[cc_gmesh]));	/*log*/
	cc_gmesh_val0 = gn_meshgc;
	fprintf(fpl,"core_correction: gmesh_val0 = meshgc\n");	/*log*/
	cc_gmesh_val1 = gn_gmaxpc;	/*fixed*/
	fprintf(fpl,"core_correction: gmesh_val1 = gmaxpc\n");	/*log*/
	cc_gmesh_val2 = 40.0;
	fprintf(fpl,"core_correction: gmesh_val2 = 40.0 : fixed\n");	/*log*/

	is_calc_ft = 1;	/*fixed*/
	fprintf(fpl,"is_calc_ft = YES : fixed\n");	/*log*/
	ft_gmesh = 0;	/*fixed*/
	fprintf(fpl,"fourier transform: gmesh = %s : fixed\n", constout(m_or_a[ft_gmesh]));	/*log*/
	ft_gmesh_val0 = 301;	/*fixed*/
	fprintf(fpl,"fourier transform: gmesh_val0 = 301 : fixed\n");	/*log*/
	ft_gmesh_val1 = 0.05;	/*fixed*/
	fprintf(fpl,"fourier transform: gmesh_val1 = 0.05 : fixed\n");	/*log*/
	ft_gmesh_val2 = 15.0;	/*fixed*/
	fprintf(fpl,"fourier transform: gmesh_val2 = 15.0 : fixed\n");	/*log*/

	is_calc_ld = 1; /*fixed*/
	fprintf(fpl,"is_calc_ld = YES : fixed\n");	/*log*/
	ld_emesh = 0;	/*fixed*/
	fprintf(fpl,"logarithmic derivative: emesh = %s : fixed\n", constout(m_or_a[ld_emesh]));	/*log*/
	ld_emesh_val0 = 601;	/*fixed*/
	fprintf(fpl,"logarithmic derivative: ld_emesh_val0 = 601 : fixed\n");	/*log*/
	ld_emesh_val1 = -2.00;	/*fixed*/
	fprintf(fpl,"logarithmic derivative: ld_emesh_val1 = -2.00 : fixed\n");	/*log*/
	ld_emesh_val2 = 1.00;	/*fixed*/
	fprintf(fpl,"logarithmic derivative: ld_emesh_val2 = 1.00 : fixed\n");	/*log*/
	ld_rcut = 2.7;	/*fixed*/
	fprintf(fpl,"logarithmic derivative: ld_rcut = 2.7 : fixed\n");	/*log*/

	num_output_pp = 1;	/*fixed*/
	output_pp[0] = 1;	/*fixed gncpp*/
	fprintf(fpl,"output_pp = %s : fixed\n", constout(output_pp_alt[output_pp[0]]));	/*log*/
	output_ft = 1;	/*fixed igor*/
	fprintf(fpl,"output_ft = %s : fixed\n", constout(output_alt[output_ft]));	/*log*/
	output_logderi = 1;	/*fixed igor*/
	fprintf(fpl,"output_logderi = %s : fixed\n", constout(output_alt[output_logderi]));	/*log*/

	is_write_debug = 0;	/*fixed*/
	fprintf(fpl,"is_write_debug = NO : fixed\n");	/*log*/

	fprintf(fpl,"\n");	/*log*/
	fprintf(fpl,"rinf: ignored in 'atom' format\n");	/*log*/

	fprintf(fpl,"}\n");
	return( 0 );
}
/************************************************************************/

/********************************************
writing gncpp format
********************************************/
int write_gncpp_form( char *fname )
{
int i, j, n, l, jc, nljc, lpsmax, ivan, fg, norb, irela;
double crc0;
int orblist[20];
double uplist[20];
double downlist[20];
int fglist[20];
int numorb;
char ebuf[64], pbuf[64];

	fprintf(fpl,"\n");
	fprintf(fpl,"WRITING{ATOM(internal)--->GNCPP}{\n");

	if( (fpw = fopen( fname, "w+" )) == NULL ){
		printf("Can't open write-file [%s] !!!\n",fname);
		fprintf(fpl,"Can't open write-file [%s] !!!\n",fname);
		return( -1 );
	}

	/********  write to screen   *******************/
	/*write_test(0);

	/********  write to file   *******************/
	numorb = 0;
	for( i = 0; i < num_cnf; i++ ){
		if( conf[i].orbital == wf_l[0].orbital || conf[i].orbital == wf_l[1].orbital
     	 || conf[i].orbital == wf_l[2].orbital || conf[i].orbital == wf_l[3].orbital ){
			fg = 1;
		}
		else{
			fg = 0;
		}
		if( conf[i].up > 0.0 || conf[i].down > 0.0 || fg){
			orblist[numorb] = conf[i].orbital;
			uplist[numorb] = conf[i].up;
			downlist[numorb] = conf[i].down;
			fglist[numorb] = fg;
			numorb++;
		}
	}

	/*fprintf(fpw,"%d  %s  %s  200.00",rmesh_val0,dtoa(ebuf,(1-rmesh_val0)/log(rmesh_val1/rmesh_val2)),dtoa(pbuf,rmesh_val2));*/	/*now fixed*/
	fprintf(fpw," %8d %14e %14e   200.000000",rmesh_val0,(1-rmesh_val0)/log(rmesh_val1/rmesh_val2),rmesh_val2);	/*now fixed*/
	fprintf(fpw," : MESH, XH, RMAX, RINF\n");
	fprintf(fpl,"mesh = rmesh_val0\n");	/*log*/
	fprintf(fpl,"xh = (1-rmesh_val0)/log(rmesh_val1/rmesh_val2)\n");	/*log*/
	fprintf(fpl,"rmax = rmesh_val2\n");	/*log*/
	fprintf(fpl,"rinf = 200.000000 : fixed\n");	/*log*/
	/*dtoa( ebuf, conv_scf_e );*/
	/*dtoa( pbuf, conv_scf_p );*/
	/*fprintf(fpw,"%lf  %s  %s        ",mix, ebuf, pbuf);*/
	fprintf(fpw," %8lf %14e %14e             ",mix, conv_scf_e, conv_scf_p);
	fprintf(fpw," : PMIX, EDEL, VDEL\n");
	fprintf(fpl,"pmix = mix\n");	/*log*/
	fprintf(fpl,"edel = conv_scf_e\n");	/*log*/
	fprintf(fpl,"vdel = conv_scf_p\n");	/*log*/

	irela = (ks_equation == 1)? 1: 2;
	fprintf(fpw,"    1    0 %4d %4d %4d                           ", spin ,numorb,irela);	/*now fixed*/
	fprintf(fpw," : JPRT, JRH, IFP, NSHL, IRELA\n");
	fprintf(fpl,"jprt = 1 : fixed\n");	/*log*/
	fprintf(fpl,"jrh = 0 : fixed\n");	/*log*/
	fprintf(fpl,"ifp = spin\n");	/*log*/
	fprintf(fpl,"nshl: number of orbitals of up > 0, down > 0, or written in wf\n");	/*log*/
	if( ks_equation == 1 ){
		fprintf(fpl,"irela: SREL--->SREL\n");	/*log*/
	}
	else{
		fprintf(fpl,"irela: %s--->SREL_GNCPP\n",constout(ks_equation_alt[ks_equation]));	/*log*/
	}

	fprintf(fpw,"%-52s", gn_xc_alt[xc_potential] );
	fprintf(fpw," : PZ, LDAPW91, GGAPW91, GGABP\n");
	strcpy( logbuf1, xc_potential_alt[xc_potential] );
	strcpy( logbuf2, gn_xc_alt[xc_potential] );
	strupper( logbuf1 );
	strupper( logbuf2 );
	fprintf(fpl,"xc_alt: %s--->%s\n", logbuf1, logbuf2 );	/*log*/
	fprintf(fpw," %4d          0.0                                  ", atom );
	fprintf(fpw," : NATOM, XION\n");
	fprintf(fpl,"natom = atom\n");	/*log*/
	fprintf(fpl,"xion = 0.0 : fixed\n");	/*log*/

	for( i = 0; i < numorb; i++ ){
		n = nl[orblist[i]][0];
		l = nl[orblist[i]][1];
		if( (uplist[i] > 0.0 || downlist[i] > 0.0) && fglist[i] ){
			jc = 5;
		}
		else if( fglist[i] ){
			jc = 6;
		}
		else{
			jc = 0;
		}
		nljc = n*100+l*10+jc;
		fprintf(fpw," %4d %12lf %12lf                     ", nljc, uplist[i], downlist[i] );
		fprintf(fpw," : NLJC, WNLJ\n", nljc, uplist[i], downlist[i] );
	}
	fprintf(fpl,"nljc: n, l from orbital, jc=0 for core, jc=5 for occupied valence, jc=6 for unoccupied valence\n");	/*log
	fprintf(fpl,"wnlj = up\n");	/*log*/
	fprintf(fpl,"down = down\n");	/*log*/

	fprintf(fpw,"*-----*-----*\n");
	for( i = 3; i >= 0; i-- ){
		if( wf_l[i].orbital >= 0 ){
			break;
		}
	}
	lpsmax = i+1;
	fprintf(fpl,"lpsmax: number of wf orbitals\n");	/*log*/

	ivan = (wf_l[0].pptype+wf_l[1].pptype+wf_l[2].pptype+wf_l[3].pptype)? 1: 0;
	fprintf(fpl,"ivan: 1 for us included, 0 for all nc\n");	/*log*/
	if( local == 0 ){	/*orbital*/
		fprintf(fpw," %4d %4d %4d                                     ", lpsmax, local_val+1, ivan);
		fprintf(fpw," : LPSMAX, ILOC, IVAN\n");
		fprintf(fpl,"iloc=local_val+1 for local=ORBITAL\n");	/*log*/
	}
	else if( local == 1 ){		/*gncpp*/
		fprintf(fpw," %4d    6 %4d %12lf %12lf %12lf", lpsmax, ivan, local_rcloc, local_const, local_lamda );
		fprintf(fpw," : LPSMAX, ILOC, IVAN, RCLOC, CONST, LAMDA\n");
		fprintf(fpl,"iloc = 6 : fixed for local=GNCPP\n");	/*log*/
		fprintf(fpl,"rcloc = local_rcloc\n");	/*log*/
		fprintf(fpl,"const = local_const\n");	/*log*/
		fprintf(fpl,"lamda = local_lamda\n");	/*log*/
	}
	else{	/* psv, forced to orbital */
		printf("'psv' is specified for 'local' parameter, forced to 'orbital'\n");
		fprintf(fpl,"'psv' is specified for 'local' parameter, forced to 'orbital'\n");
		fprintf(fpw," %4d %4d %4d                                     ", lpsmax, local_val+1, ivan);
		fprintf(fpw," : LPSMAX, ILOC, IVAN\n");
		fprintf(fpl,"iloc=local_val+1 for local=ORBITAL\n");	/*log*/
	}

	for( j = 0; j < 4; j++ ){
		if( wf_l[j].orbital >= 0 ){
			fprintf(fpw," %4d %4d %4d                                     ", j+1, wf_l[j].num_rcut, wf_l[j].pptype);
			fprintf(fpw," : l+1, itau, ivanl\n");
			for( norb = 0; norb < numorb; norb++ ){
				if( orblist[norb] == wf_l[j].orbital ){
					break;
				}
			}
			norb++;
			for( i = 0; i < wf_l[j].num_rcut; i++ ){
				fprintf(fpw,"                %4d %12lf %12lf      ", norb, wf_l[j].rcut[i][0], wf_l[j].rcut[i][1]);
				fprintf(fpw," : norb, r_cl, etau\n");
			}
		}
	}
	fprintf(fpl,"l+1: from wf orbital\n");	/*log*/
	fprintf(fpl,"itau: number of rcut\n");	/*log*/
	fprintf(fpl,"ivanl: 1 for us, 0 for nc\n");	/*log*/
	fprintf(fpl,"norb: place in nljc list\n");	/*log*/
	fprintf(fpl,"r_cl: from wf rcut0\n");	/*log*/
	fprintf(fpl,"etau: from wf rcut1\n");	/*log*/
	
	fprintf(fpw," %12lf %12d %12d             ", wf_gmesh_val1, wf_gmesh_val0, ((wf_nc_shape_val > wf_us_shape_val)? wf_nc_shape_val-1: wf_us_shape_val-1) );
	fprintf(fpw," : GMAXWF, MESHGW, MOPSW\n");
	fprintf(fpl,"gmaxwf = wf_gmesh_val1\n");	/*log*/
	fprintf(fpl,"meshgw = wf_gmesh_val0\n");	/*log*/
	fprintf(fpl,"mopsw = (larger of wf_nc_shape_val and wf_us_shape_val)-1\n");	/*log*/
	fprintf(fpw," %12lf %12d %12d             ", qps_gmesh_val1, qps_gmesh_val0, qps_shape_val-1);
	fprintf(fpw," : GMAXPS, MESHGS, MOPSC\n");
	fprintf(fpl,"gmaxps = qps_gmesh_val1\n");	/*log*/
	fprintf(fpl,"meshgs = qps_gmesh_val0\n");	/*log*/
	fprintf(fpl,"mopsc = qps_shape_val-1\n");	/*log*/

	for( i = 0; i < qps_num_rcut; i++ ){
		fg = (i == qps_num_rcut-1)? 0: 1;
		fprintf(fpw," %4d %4d %4d %4d %4d %12lf %4d         ",qps_rcut[i][0]+1,qps_rcut[i][1]+1,qps_rcut[i][2]+1,qps_rcut[i][3]+1,qps_rcut[i][4]+1,qps_rcut_val[i],fg);
		fprintf(fpw," : IL1,T1,IL2,T2,IL3,RCMLT,IL4\n");
	}
	fprintf(fpl,"il1 = qps_rcut0+1\n");	/*log*/
	fprintf(fpl,"t1 = qps_rcut1+1\n");	/*log*/
	fprintf(fpl,"il2 = qps_rcut2+1\n");	/*log*/
	fprintf(fpl,"t2 = qps_rcut3+1\n");	/*log*/
	fprintf(fpl,"il3 = qps_rcut4+1\n");	/*log*/
	fprintf(fpl,"rcmlt = qps_rcut_val\n");	/*log*/
	fprintf(fpl,"il4: 0 for the last one, 1 for the others\n");	/*log*/

	fprintf(fpw," %4d                                               ", core_corre );
	fprintf(fpw," : IPCC\n");
	fprintf(fpl,"ipcc = core_corre\n");	/*log*/
	if( core_corre == 1 ){
		crc0 = (cc_rcut <= 1)? -cc_rcut_val: cc_rcut_val;
		fprintf(fpw," %12lf                                       ",crc0);
		fprintf(fpw," : CRCO\n");
		if( cc_rcut <= 1 ){
			fprintf(fpl,"crc0 = -cc_rcut_val for cc_rcut=AUTOMATIC or RADIUS\n");	/*log*/
		}
		else{
			fprintf(fpl,"crc0 = cc_rcut_val for cc_rcut=CV_RATIO\n");	/*log*/
		}
	}

	fprintf(fpw," %12lf %12d %12d             ", cc_gmesh_val2, cc_gmesh_val0, cc_shape_val-1);
	fprintf(fpw," : GMAXPC, MESHGC, MOPCC\n");
	fprintf(fpl,"gmaxpc = cc_gmesh_val2\n");	/*log*/
	fprintf(fpl,"meshgc = cc_gmesh_val0\n");	/*log*/
	fprintf(fpl,"mopcc = cc_shape_val-1\n");	/*log*/
	fclose(fpw);

	fprintf(fpl,"\n");	/*log*/
	fprintf(fpl,"spin_val: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"core_potential: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"anderson: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"conv_max_iteration: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"conv_wf: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"pre_cor0: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"pre_cor1: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"diff_order: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"num_coeff_veff: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"output_rho: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"output_rm: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"is_calc_pp: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"wf_nc_shape: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"wf_us_shape: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"wf_gmesh_val1: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"local: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"qps_shape: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"qps_gmesh_val1: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"core_corre_shape: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"core_corre_gmesh_val1: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"is_calc_ft: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"ft_gmesh_val0: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"ft_gmesh_val1: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"ft_gmesh_val2: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"is_calc_ld: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"ld_gmesh_val0: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"ld_gmesh_val1: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"ld_gmesh_val2: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"output_pp: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"output_ft: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"output_logderi: ignored in 'gncpp' format\n");	/*log*/
	fprintf(fpl,"is_write_debug: ignored in 'gncpp' format\n");	/*log*/

	fprintf(fpl,"}\n");
	return( 0 );
}

/************************************************************************************/

/********************************************
psv format reading functions
********************************************/

int psv_skip_comment( char *buf, int p, int imax )
{
int i;

	i = p;
	while( i < imax ){
		get_parastr( pbuf, buf, &i, imax );
		if( strcmp( pbuf, "END-COMMENT" ) == 0 ){
			break;
		}
		if( buf[i] == '#' ){
			i = gotonextline( buf, i, imax );
			break;
		}
	}
	if( i >= imax ){
		i = imax;
	}
	return( i );
}


int get_psv_allel()
{
char deb[16];
int i;

	rptr = gotonextline( rbuf, rptr, rlen );
	rptr = gotonextline( rbuf, rptr, rlen );	/*skip title*/

	get_parastr( pbuf, rbuf, &rptr, rlen );
	for( i = 0; pbuf[i] != 0; i++ ){
		pbuf[i] = tolower( pbuf[i] );
	}
	for( i = 0; i < NPSVXCALT; i++ ){
		if( strcmp( pbuf, psv_xc_alt[i] ) == 0 ){
			break;
		}
	}
	if( i >= NPSVXCALT ){
		psv_xctype = -1;
		printf("wrong XC name: %s !!!\n", pbuf );
		fprintf(fpl,"wrong XC name: %s !!!\n", pbuf);
		return( -1 );
	}
	else{
		psv_xctype = i;
	}
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_zn = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_irela = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_ncst = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );
	for( i = 0; i < psv_ncst; i++ ){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_ce[i].n = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_ce[i].l = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_ce[i].occup = atof( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_ce[i].e = atof( pbuf );
		rptr = gotonextline( rbuf, rptr, rlen );
	}
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_nvstal = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );
	for( i = 0; i < psv_nvstal; i++ ){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_ve[i].n = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_ve[i].l = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_ve[i].occup = atof( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_ve[i].e = atof( pbuf );
		rptr = gotonextline( rbuf, rptr, rlen );
	}

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_itermx = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_mesh = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_radmin = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_radmax = atof( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_conv = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_beta = atof( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	strupper( pbuf );
	if( strcmp( pbuf, "END-ALLEL" ) != 0 ){
		printf("END-ALLEL not found or too many/few parameters !!!\n");
		fprintf(fpl,"END-ALLEL not found or too many/few parameters !!!\n");
		/*scanf("%s", deb);*/
		return( -1 );
	}

	return( 0 );
}

int get_psv_psv()
{
char deb[16];
int i, j, nvst;

/*printf("come get_psv_psv\n");*/
	rptr = gotonextline( rbuf, rptr, rlen );
	rptr = gotonextline( rbuf, rptr, rlen );	/*skip title*/

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_zv = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_ipcc = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_rpcc = atof( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	nvst = psv_nvst = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );
	for( i = 0; i < psv_nvst; i++ ){
		/*printf("nvst=%d\n",psv_nvst);*/
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_wf[i].nv = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_wf[i].lv = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_wf[i].rc = atof( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_wf[i].ipot = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_wf[i].iwav = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_wf[i].nref = atoi( pbuf );
		for( j = 0; j < psv_wf[i].nref; j++ ){
			/*printf("nref=%d\n",psv_wf[i].nref);*/
			get_parastr( pbuf, rbuf, &rptr, rlen );
			psv_wf[i].eref[j] = atof( pbuf );
		}
		rptr = gotonextline( rbuf, rptr, rlen );
	}

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_iloc = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_np = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_lp = atoi( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_rcp = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_iwavp = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_ipq = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	/****
	for( i = 0; i < (nvst*(nvst+1))/2; i++ ){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_rin[i] = atof( pbuf );
	}
	*****/
	i = 0;
	while(1){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		if( pbuf[0] >= '0' && pbuf[0] <= '9' ){ 
			psv_rin[i] = atof( pbuf );
			i++;
		}
		else{
			break;
		}
	}
	for( ; i < 7; i++ ){
		psv_rin[i] = 1.0;
	}
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_gc = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_gh = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_ng = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_nsst = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	for( i = 0; i < psv_nsst; i++ ){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_ns[i].ns = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_ns[i].ls = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_ns[i].swgt = atof( pbuf );
		rptr = gotonextline( rbuf, rptr, rlen );
	}

	get_parastr( pbuf, rbuf, &rptr, rlen );
	strupper( pbuf );
	if( strcmp( pbuf, "END-PSV" ) != 0 ){
		printf("END-PSV not found or too many/few parameters !!!\n");
		fprintf(fpl,"END-PSV not found or too many/few parameters !!!\n");
		/*scanf("%s", deb);*/
		return( -1 );
	}

	/*printf("end get_psv_psv\n");*/
	return( 0 );
}

int get_psv_solps()
{
int ret = 0, i;
char deb[16];

	rptr = gotonextline( rbuf, rptr, rlen );
	rptr = gotonextline( rbuf, rptr, rlen );	/*skip title*/

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_norb = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );
	for( i = 0; i < psv_norb; i++ ){
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_solps[i].n = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_solps[i].l = atoi( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_solps[i].occup = atof( pbuf );
		get_parastr( pbuf, rbuf, &rptr, rlen );
		psv_solps[i].e = atof( pbuf );
		rptr = gotonextline( rbuf, rptr, rlen );
	}

	get_parastr( pbuf, rbuf, &rptr, rlen );
	strupper( pbuf );
	if( strcmp( pbuf, "END-SOLPS" ) != 0 ){
		printf("END-SOLPS not found or too many/few parameters !!!\n");
		fprintf(fpl,"END-SOLPS not found or too many/few parameters !!!\n");
		/*scanf("%s", deb);*/
		return( -1 );
	}

	return( ret );
}

int get_psv_drv()
{
int i;
char deb[16];

	rptr = gotonextline( rbuf, rptr, rlen );
	rptr = gotonextline( rbuf, rptr, rlen );	/*skip title*/

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_emin = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_emax = atof( pbuf );
	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_neng = atoi( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	psv_rdrv = atof( pbuf );
	rptr = gotonextline( rbuf, rptr, rlen );

	get_parastr( pbuf, rbuf, &rptr, rlen );
	for( i = 0; pbuf[i] != 0; i++ ){
		pbuf[i] = toupper( pbuf[i] );
	}
	if( strcmp( pbuf, "END-DRV" ) != 0 ){
		printf("END-DRV not found or too many/few parameters !!!\n");
		fprintf(fpl,"END-DRV not found or too many/few parameters !!!\n");
		/*scanf("%s", deb);*/
		return( -1 );
	}

	return( 0 );
}

int read_psv_form( char *fname )
{
int     ret = 0, i, j, l1, t1, l2, t2, l3, l1max, l2max, l3max, tmax[4], ir;
int     ddd, errfg = 0;
char deb[16];

	fprintf(fpl,"READING{PSV--->ATOM(internal)}{\n");

	if( (fpr = fopen( fname, "r" )) == NULL ){
		printf("Can't open input file [%s] !!!\n",fname);
		return( -1 );
	}

	rptr = 0;
	fseek( fpr, 0, SEEK_SET );
	rlen = fread( rbuf, 1, RBUFSIZE, fpr );
	fclose( fpr );
	if( rlen < 0 ){
		printf("read error.\n");
		return( rlen );
	}

	while( rptr < rlen ){
		if( get_parastr( pbuf, rbuf, &rptr, rlen ) < 0 ){
			break;
		}
		/***
		ddd = strlen(pbuf);
		printf("pbuf=[%s] len=%d\n",pbuf, ddd);
		ddd = strcmp( pbuf, "BEGIN-ALLEL" );
		printf("strcmp=%d\n",ddd);
		***/
		if( strcmp( pbuf, "BEGIN-ALLEL" ) == 0 ){
			if( (errfg=get_psv_allel()) < 0 ){
				break;
			}
		}
		else if( strcmp( pbuf, "BEGIN-PSV" ) == 0 ){
			if( (errfg=get_psv_psv()) < 0 ){
				break;
			}
		}
		else if( strcmp( pbuf, "BEGIN-SOLPS" ) == 0 ){
			if( (errfg=get_psv_solps()) < 0 ){
				break;
			}
		}
		else if( strcmp( pbuf, "BEGIN-DRV" ) == 0 ){
			if( (errfg=get_psv_drv()) < 0 ){
				break;
			}
		}
		else if( strcmp( pbuf, "BEGIN-COMMENT" ) == 0 ){
			rptr = psv_skip_comment( rbuf, rptr, rlen );
		}
		else{
			errfg = 1;
			printf("illeagal keyword found !!!\n");
			fprintf(fpl,"illeagal keyword found !!!\n");
			/*scanf("%s", deb );*/
			break;
		}
	}

	if( errfg ){
		return(-1);
	}

/*printf("end reading\n");*/
	/*fprintf(fpl,"}\n");*/
	/*fprintf(fpl,"CONVERSION{\n");*/

	/**** put psv variable to std. variable ****/
	atom = (int)psv_zn;
	fprintf(fpl,"atom = zn\n");	/*log*/
	sprintf( title, "atom_%d", atom );
	spin = 0;		/*fixed, spin polarization not supported by psv*/
	fprintf(fpl,"spin=0 (non-polarized) : fixed\n");	/*log*/
	spin_val = 0.0;	/*fixed*/
	fprintf(fpl,"spin_val = 0.0 : fixed\n");	/*log*/
	ks_equation = psv_irela;
	fprintf(fpl,"ks_equation = irela\n");	/*log*/
	num_cnf = psv_ncst+psv_nvstal;
	fprintf(fpl,"configuration: number of orbitals=ncst+nvstal\n");	/*log*/
	for( i = 0; i < psv_ncst; i++ ){
		for( j = 0; j < 19; j++ ){
			if( nl[j][0] == psv_ce[i].n && nl[j][1] == psv_ce[i].l ){
				break;
			}
		}
		conf[i].orbital = j;
		conf[i].up = psv_ce[i].occup;
		conf[i].down = 0.0;				/*fixed*/
	}
	for( i = 0; i < psv_nvstal; i++ ){
		for( j = 0; j < 19; j++ ){
			if( nl[j][0] == psv_ve[i].n && nl[j][1] == psv_ve[i].l ){
				break;
			}
		}
		conf[i+psv_ncst].orbital = j;
		conf[i+psv_ncst].up = psv_ve[i].occup;
		conf[i+psv_ncst].down = 0.0;		/*fixed*/
	}
	fprintf(fpl,"configuration: orbital from nc, lc, nv, lv\n");	/*log*/
	fprintf(fpl,"configuration: up = occupc, occupv\n");	/*log*/
	fprintf(fpl,"configuration: down = 0.0 : fixed\n");	/*log*/
	core_potential = 0;	/*fixed*/
	fprintf(fpl,"core_potential=NORMAL : fixed\n");	/*log*/
	if( psv_xctype == 0 || psv_xctype == 2 ){	/*GGAPBE96, GGAPW91 -> GGAPBE96*/
		xc_potential = 2;
	}
	else{										/*LDAPW92 -> LDAPW92*/
		xc_potential = 1;
	}
	strcpy( logbuf1, psv_xc_alt[psv_xctype] );
	strcpy( logbuf2, xc_potential_alt[xc_potential] );
	strupper( logbuf1 );
	strupper( logbuf2 );
	fprintf(fpl,"xc_potential: %s--->%s\n", logbuf1, logbuf2 );	/*log*/
	rmesh = 0;	/*fixed, manual*/
	fprintf(fpl,"rmesh = MANUAL : fixed\n" );	/*log*/
	rmesh_val0 = psv_mesh;
	fprintf(fpl,"rmesh_val0 = mesh\n" );	/*log*/
	rmesh_val1 = psv_radmin;
	fprintf(fpl,"rmesh_val1 = rad(1)\n" );	/*log*/
	rmesh_val2 = psv_radmax;
	fprintf(fpl,"rmesh_val2 = rad(mesh)\n" );	/*log*/
	mix = psv_beta;
	fprintf(fpl,"mix = beta\n" );	/*log*/
	anderson = -1;	/*fixed*/
	fprintf(fpl,"anderson: not used : fixed\n" );	/*log*/
	conv_max_iteration = psv_itermx;
	fprintf(fpl,"conv_max_iteration = itermx\n" );	/*log*/
	/*conv_scf_e = psv_conv;*/	/*assumed*/
	conv_scf_e = 1.0e-8;	/*fixed*/
	fprintf(fpl,"conv_scf_e = conv : assumed\n" );	/*log*/
	/*conv_scf_p = psv_conv;*/
	conv_scf_p = 1.0e-8;	/*fixed*/
	fprintf(fpl,"conv_scf_p = conv\n" );	/*log*/
	conv_wf = 1.0e-10;	/*fixed*/
	fprintf(fpl,"conv_wf = 1.0e-10 : fixed\n" );	/*log*/
	pre_cor0 = 5;	/*fixed*/
	fprintf(fpl,"pre_cor0 = 5 : fixed\n" );	/*log*/
	pre_cor1 = 3;	/*fixed*/
	fprintf(fpl,"pre_cor1 = 3 : fixed\n" );	/*log*/
	diff_order = 4;	/*fixed*/
	fprintf(fpl,"diff_order = 4 : fixed\n" );	/*log*/
	num_coeff_veff = 5;	/*fixed*/
	fprintf(fpl,"num_coeff_veff = 5 : fixed\n" );	/*log*/
	output_rho = 1;	/*fixed igor*/
	fprintf(fpl,"output_rho = IGOR : fixed\n" );	/*log*/
	output_rm = 1;	/*fixed*/
	fprintf(fpl,"output_rm = IGOR : fixed\n" );	/*log*/

	is_calc_pp = 1; /*fixed*/
	fprintf(fpl,"is_calc_pp = YES : fixed\n" );	/*log*/
	wf_nc_shape = -1;
	wf_us_shape = -1;
	for( i = 0; i < 4; i++ ){
		if( i >= psv_nvst ){
			wf_l[i].orbital = -1;
			wf_l[i].pptype = 0;
			continue;
		}
		for( j = 0; j < 19; j++ ){
			if( nl[j][0] == psv_wf[i].nv && nl[j][1] == psv_wf[i].lv ){
				break;
			}
		}
		wf_l[i].orbital = j;
		wf_l[i].pptype = !(psv_wf[i].ipot);
		if( wf_l[i].pptype == 0 ){	/*norm conserving*/
			if( psv_wf[i].iwav == 0 ){	/*T-M*/
				wf_nc_shape = 2;
			}
			else{						/*Kerker*/
				wf_nc_shape = 3;
			}
			wf_nc_shape_val = 1; /*fixed*/
		}
		else{						/*ultrasoft*/
			wf_us_shape = 0;	/*fixed*/
			wf_us_shape_val = 10;	/*fixed*/
		}
		wf_l[i].num_rcut = psv_wf[i].nref;
		tmax[i] = psv_wf[i].nref;
		for( j = 0; j < wf_l[i].num_rcut; j++ ){
			wf_l[i].rcut[j][0] = psv_wf[i].rc;
			wf_l[i].rcut[j][1] = psv_wf[i].eref[j];
		}
	}
	fprintf(fpl,"wf: number of orbitals = nvst\n" );	/*log*/
	fprintf(fpl,"wf: orbital from nv, lv\n" );	/*log*/
	fprintf(fpl,"wf: pptype = !(ipot)\n" );	/*log*/
	fprintf(fpl,"wf: number of rcut = nref\n" );	/*log*/
	fprintf(fpl,"wf: rcut0 = rc\n" );	/*log*/
	fprintf(fpl,"wf: rcut1 = eref\n" );	/*log*/
	fprintf(fpl,"wf: nc_shape = TM90 for iwav=0, KERKER85 for iwav=1\n" );	/*log*/
	fprintf(fpl,"wf: nc_shape_val = 1 : fixed\n" );	/*log*/
	fprintf(fpl,"wf: us_shape = POLYNOMIAL : fixed\n" );	/*log*/
	fprintf(fpl,"wf: us_shape_val = 10 : fixed\n" );	/*log*/

	wf_gmesh = 0;	/*fixed, manual*/
	fprintf(fpl,"wf: gmesh = MANUAL : fixed\n" );	/*log*/
	/*wf_gmesh_val0 = psv_ng;*/
	wf_gmesh_val0 = 401;
	/*fprintf(fpl,"wf: gmesh_val0 = ng\n" );*/	/*log*/
	fprintf(fpl,"wf: gmesh_val0 = 401 : fixed\n" );	/*log*/
	/*wf_gmesh_val1 = psv_gc;*/
	wf_gmesh_val1 = 5.0;
	/*fprintf(fpl,"wf: gmesh_val1 = gc\n" );*/	/*log*/
	fprintf(fpl,"wf: gmesh_val1 = 5.0 : fixed\n" );	/*log*/
	/*wf_gmesh_val2 = psv_gc+(psv_ng-1)*psv_gh;*/
	wf_gmesh_val2 = 40.0;
	/*fprintf(fpl,"wf: gmesh_val2 = gc+(ng-1)*gh\n" );*/	/*log*/
	fprintf(fpl,"wf: gmesh_val2 = 40.0 : fixed\n" );	/*log*/

	if( psv_iloc == 0 ){
		local = 0;	/*fixed, orbital*/
		fprintf(fpl,"local = ORBITAL : for iloc = 0\n" );	/*log*/
	}
	else{
		local = 2;	/*fixed, psv*/
		fprintf(fpl,"local = PSV : for iloc != 0\n" );	/*log*/
	}
	local_val = psv_nvst-1;
	fprintf(fpl,"local_val = nvst\n" );	/*log*/
	l1max = psv_nvst-1;
	l2max = psv_nvst-1;
	ir = 0;
	qps_num_rcut = 0;
	for( l1 = 0; l1 <= l1max; l1++ ){
		if( l1 == local_val ){
			continue;
		}
		for( l2 = 0; l2 <= l2max; l2++ ){
			if( l1 > l2  || l2 == local_val ){
				continue;
			}
			for( t1 = 0; t1 < tmax[l1]; t1++ ){
				for( t2 = 0; t2 < tmax[l2]; t2++ ){
					if( t1 > t2 ){
						continue;
					}
					l3 = l1-l2;
					if( l3 < 0 ){
						l3 *= -1;
					}
					l3max = l1+l2;
					for( ; l3 <= l3max;  ){
						qps_rcut[qps_num_rcut][0] = l1;
						qps_rcut[qps_num_rcut][1] = t1+1;
						qps_rcut[qps_num_rcut][2] = l2;
						qps_rcut[qps_num_rcut][3] = t2+1;
						qps_rcut[qps_num_rcut][4] = l3;
						/*qps_rcut_val[qps_num_rcut] = psv_rin[ir];*/
						qps_rcut_val[qps_num_rcut] = psv_rin[l3];
						qps_num_rcut++;
						l3 += 2;
					}
				}
			}
			ir++;
		}
	}
	fprintf(fpl,"qps: rcut0,rcut1,rcut2,rcut3,rcut4 = all possible combinations are listed up\n" );	/*log*/
	fprintf(fpl,"qps: rcut_val = rin : different rin for different (il1,il2)\n" );	/*log*/
	qps_shape = 0;	/*fixed*/
	fprintf(fpl,"qps: shape = POLYNOMIAL : fixed\n" );	/*log*/
	qps_shape_val = 10;	/*fixed*/
	fprintf(fpl,"qps: shape_val = 10 : fixed\n" );	/*log*/

	qps_gmesh = 0;	/*fixed, manual*/
	fprintf(fpl,"qps: gmesh = MANUAL : fixed\n" );	/*log*/
	/*qps_gmesh_val0 = 401;*/
	qps_gmesh_val0 = psv_ng;
	/*fprintf(fpl,"qps: gmesh_val0 = 401 : fixed\n" );*/	/*log*/
	fprintf(fpl,"qps: gmesh_val0 = psv_ng\n" );	/*log*/
	/*qps_gmesh_val1 = 12.0;*/
	qps_gmesh_val1 = psv_gc;
	/*fprintf(fpl,"qps: gmesh_val1 = 12.0 : fixed\n" );*/	/*log*/
	fprintf(fpl,"qps: gmesh_val1 = psv_gc\n" );	/*log*/
	/*qps_gmesh_val2 = 40.0;*/
	qps_gmesh_val2 = psv_gc+(psv_ng-1)*psv_gh;
	/*fprintf(fpl,"qps: gmesh_val2 = 40.0 : fixed\n" );*/	/*log*/
	fprintf(fpl,"qps: gmesh_val2 = psv_gc+(psv_ng-1)*psv_gh;\n" );	/*log*/


	core_corre = psv_ipcc;
	fprintf(fpl,"core_correction = ipcc\n" );	/*log*/
	cc_rcut = 1;	/*fixed, radius*/
	fprintf(fpl,"core_correction: rcut = RADIUS : fixed\n" );	/*log*/
	cc_rcut_val = psv_rpcc;
	fprintf(fpl,"core_correction: rcut_val = rpcc\n" );	/*log*/
	cc_shape = 0;	/*fixed*/
	fprintf(fpl,"core_correction: shape = POLYNOMIAL : fixed\n" );	/*log*/
	cc_shape_val = 10;	/*fixed*/
	fprintf(fpl,"core_correction: shape_val = 10 : fixed\n" );	/*log*/
	cc_gmesh = 0;	/*fixed*/
	fprintf(fpl,"core_correction: gmesh = MANUAL : fixed\n" );	/*log*/
	cc_gmesh_val0 = 1001;	/*fixed*/
	fprintf(fpl,"core_correction: gmesh_val0 = 1001 : fixed\n" );	/*log*/
	cc_gmesh_val1 = 12.0;	/*fixed*/
	fprintf(fpl,"core_correction: gmesh_val1 = 12.0 : fixed\n" );	/*log*/
	cc_gmesh_val2 = 40.0;	/*fixed*/
	fprintf(fpl,"core_correction: gmesh_val2 = 40.0 : fixed\n" );	/*log*/

	is_calc_ft = 1;	/*fixed*/
	fprintf(fpl,"is_calc_ft = YES : fixed\n" );	/*log*/
	ft_gmesh = 0;	/*fixed*/
	fprintf(fpl,"fourier transform: gmesh = MANUAL : fixed\n" );	/*log*/
	ft_gmesh_val0 = 301;	/*fixed*/
	fprintf(fpl,"fourier transform: gmesh_val0 = 301 : fixed\n" );	/*log*/
	ft_gmesh_val1 = 0.05;	/*fixed*/
	fprintf(fpl,"fourier transform: gmesh_val1 = 0.05 : fixed\n" );	/*log*/
	ft_gmesh_val2 = 15.0;	/*fixed*/
	fprintf(fpl,"fourier transform: gmesh_val2 = 15.0 : fixed\n" );	/*log*/

	is_calc_ld = 1; /*fixed*/
	fprintf(fpl,"is_calc_ld = YES : fixed\n" );	/*log*/
	ld_emesh = 0;	/*fixed, manual*/
	fprintf(fpl,"logarithmic derivative: emesh = MANUAL : fixed\n" );	/*log*/
	ld_emesh_val0 = psv_neng;
	fprintf(fpl,"logarithmic derivative: emesh_val0 = neng\n" );	/*log*/
	ld_emesh_val1 = psv_emin;
	fprintf(fpl,"logarithmic derivative: emesh_val1 = emin\n" );	/*log*/
	ld_emesh_val2 = psv_emax;
	fprintf(fpl,"logarithmic derivative: emesh_val2 = emax\n" );	/*log*/
	ld_rcut = psv_rdrv;
	fprintf(fpl,"logarithmic derivative: rcut = rdrv\n" );	/*log*/

	num_output_pp = 1;	/*fixed*/
	output_pp[0] = 1;	/*fixed gncpp*/
	fprintf(fpl,"output_pp = GNCPP : fixed\n" );	/*log*/
	output_ft = 1;	/*fixed igor*/
	fprintf(fpl,"output_ft = IGOR : fixed\n" );	/*log*/
	output_logderi = 1;	/*fixed igor*/
	fprintf(fpl,"output_logderi = IGOR : fixed\n" );	/*log*/

	is_write_debug = 0;
	fprintf(fpl,"is_write_debug = NO : fixed\n" );	/*log*/

	fprintf(fpl,"\n" );	/*log*/
	fprintf(fpl,"ALLEL part: ec, ev ignored in 'atom' format\n" );	/*log*/
	fprintf(fpl,"PSV part: np, lp, rcp, iwavp, ipq, nsst, ns, ls, swgt ignored in 'atom' format\n" );	/*log*/
	fprintf(fpl,"SOLPS part: norb, nv, lv, occu, ein ignored in 'atom' format\n" );	/*log*/
	fprintf(fpl,"}\n");
	return( 0 );
}
/************************************************************************/

double get_init_energy( int atomno, int n, int l )
{
double ret;

	if( n == 1 ){
		ret = -65.0;
	}
	else if( n == 2 ){
		if( l == 0 ){
			ret = -5.0;
		}
		else{
			ret = -3.5;
		}
	}
	else if( n == 3 ){
		if( l == 0 ){
			ret = -0.4;
		}
		else if( l == 1 ){
			ret = -0.2;
		}
		else{
			ret = -0.1;
		}
	}
	else{
		ret = -0.1;
	}
	
	ret *= (atomno*atomno)/(14.0*14.0);
	return(ret);
}

/********************************************
writing psv format
********************************************/
int write_psv_form( char *fname )
{
int i, j, k, n, l, fg, nvst, ipq, iwav, prev_l1, prev_l2, l1max, l2max, l1, l2;
int irela;
double zv;
int c_orblist[20];
double c_uplist[20];
double c_downlist[20];
int c_numorb;
int v_orblist[20];
double v_uplist[20];
double v_downlist[20];
int v_numorb;
char ebuf[64], pbuf[64];
char deb[16];
double rin[7], rcp, einit;
int	rinfg[7], l3, rinmax, iloc;

	fprintf(fpl,"WRITING{ATOM(internal)--->PSV}{\n");

	if( (fpw = fopen( fname, "w+" )) == NULL ){
		printf("Can't open write-file [%s] !!!\n", fname );
		fprintf(fpl,"Can't open write-file [%s] !!!\n", fname );
		return( -1 );
	}

	/********  write to screen   *******************/
	/*write_test(0);*/

	/********  write to file   *******************/
	fprintf(fpw,"BEGIN-ALLEL\n");
	fprintf(fpl,"BEGIN-ALLEL\n");
	fprintf(fpw,"%s\n",title);

	if( xc_potential == 0 || xc_potential >= 3 ){
		printf("xc type not supported by psv !!!\n");
		fprintf(fpl,"xc type not supported by psv !!!\n");
		/*scanf("%s", deb);*/
		return(-1);
	}
	if( spin == 1 ){
		printf("Polarized spin not supported by psv !!!\n");
		fprintf(fpl,"Polarized spin not supported by psv !!!\n");
		/*scanf("%s", deb);*/
		return(-1);
	}
	if( ks_equation >= 2 ){
		irela = 1;
		/*printf("relativistic type not supported by psv !!!\n");*/
		fprintf(fpl,"relativistic type not supported by psv, changed to SREL\n");
		/*scanf("%s", deb);*/
		/*return(-1);*/
	}
	else{
		irela = ks_equation;
	}
	sprintf( logbuf1, "%s", xc_potential_alt[xc_potential] );
	sprintf( logbuf2, "%s", psv_xc_alt[xc_potential] );
	strupper( logbuf1 );
	strupper( logbuf2 );
	fprintf(fpl,"xctype: %s--->%s\n",logbuf1,logbuf2);
	fprintf(fpl,"zn = atom\n");
	fprintf(fpl,"irela = ks_equation\n");

	fprintf(fpw,"%-12s %4d %4d                              ", logbuf2, atom, irela );
	fprintf(fpw," # xctype zn irela\n");

	zv = 0.0;
	c_numorb = v_numorb = 0;
	for( i = 0; i < num_cnf; i++ ){
		if( conf[i].orbital == wf_l[0].orbital || conf[i].orbital == wf_l[1].orbital
     	 || conf[i].orbital == wf_l[2].orbital || conf[i].orbital == wf_l[3].orbital ){
			fg = 1;
		}
		else{
			fg = 0;
		}
		if( conf[i].up > 0.0 /*|| conf[i].down > 0.0 || fg*/){
			if( fg ){	/*valence*/
				v_orblist[v_numorb] = conf[i].orbital;
				v_uplist[v_numorb] = conf[i].up;
				v_downlist[v_numorb] = conf[i].down;
				zv += conf[i].up;
				v_numorb++;
			}
			else{		/*core*/
				c_orblist[c_numorb] = conf[i].orbital;
				c_uplist[c_numorb] = conf[i].up;
				c_downlist[c_numorb] = conf[i].down;
				c_numorb++;
			}
		}
	}

	fprintf(fpw," %4d                                               ", c_numorb );
	fprintf(fpw," # ncst\n");
	fprintf(fpl,"ncst: number of orbitals not written in wf\n");
	for( i = 0; i < c_numorb; i++ ){
		n = nl[c_orblist[i]][0];
		l = nl[c_orblist[i]][1];
		einit = get_init_energy( atom, n, l );
		fprintf(fpw," %4d %4d %12lf %12lf                ", n, l, c_uplist[i], einit /*-((double)atom*atom)/(2.0*n*n)*/ );
		fprintf(fpw," # nc lc occupc ec\n");
	}
	fprintf(fpl,"nc, lc: orbital not written in wf\n");
	fprintf(fpl,"occupc = up\n");
	fprintf(fpl,"ec = -(Z^2)/(2n^2) : assumed\n");
	fprintf(fpw," %4d                                               ", v_numorb );
	fprintf(fpw," # nvstal\n");
	fprintf(fpl,"nvstal: number of orbitals written in wf\n");
	for( i = 0; i < v_numorb; i++ ){
		n = nl[v_orblist[i]][0];
		l = nl[v_orblist[i]][1];
		einit = get_init_energy( atom, n, l );
		fprintf(fpw," %4d %4d %12lf %12lf                ", n, l, v_uplist[i], einit /*-((double)atom*atom)/(2.0*n*n)*/ );
		fprintf(fpw," # nv lv occupv ev\n");
	}
	fprintf(fpl,"nv, lv: orbital written in wf\n");
	fprintf(fpl,"occupv = up\n");
	fprintf(fpl,"ev = -(Z^2)/(2n^2) : assumed\n");

	/*dtoa( ebuf, rmesh_val1 );*/
	/*dtoa( pbuf, rmesh_val2 );*/
	/*fprintf(fpw,"%d  %d  %s  %s\n",  conv_max_iteration, rmesh_val0, ebuf, pbuf);*/
	fprintf(fpw," %4d %4d %14e %14e            ",  conv_max_iteration, rmesh_val0, rmesh_val1, rmesh_val2);
	fprintf(fpw," # itermx mesh rad(1) rad(meh)\n");
	fprintf(fpl,"itermx = conv_max_iteration\n");
	fprintf(fpl,"mesh = rmesh_val0\n");
	fprintf(fpl,"rad(1) = rmesh_val1\n");
	fprintf(fpl,"rad(mesh) = rmesh_val2\n");

	/*dtoa( ebuf, conv_scf_p );*/
	/*fprintf(fpw,"%s  %lf\n",  ebuf, mix);*/
	/*fprintf(fpw," %14e %12lf                        ",  conv_scf_p, mix);*/
	fprintf(fpw," 1.000000e-16 %12lf                        ",  conv_scf_p, mix);
	fprintf(fpw," # conv beta\n");
	fprintf(fpl,"conv = conv_scf_p\n");
	fprintf(fpl,"beta = mix\n");
	fprintf(fpw,"END-ALLEL\n");
	fprintf(fpl,"END-ALLEL\n");


	fprintf(fpw,"BEGIN-PSV\n");
	fprintf(fpl,"BEGIN-PSV\n");
	fprintf(fpw,"%s\n",title);
	fprintf(fpw," %12lf %4d %12lf                     ", zv, core_corre, cc_rcut_val );
	fprintf(fpw," # zv ipcc rpcc\n");
	fprintf(fpl,"zv = sum of valence 'up'\n");
	fprintf(fpl,"ipcc = core_corre\n");
	fprintf(fpl,"rpcc = cc_rcut_val\n");

	nvst = 0;
	for( i = 0; i < 4; i++ ){
		if( wf_l[i].orbital < 0 ){
			continue;
		}
		for( k = 0; k < v_numorb; k++ ){
			if( wf_l[i].orbital == v_orblist[k] ){
				break;
			}
		}
		if( k >= v_numorb ){
			continue;
		}
		nvst++;
	}
	fprintf(fpw," %4d                                               ", nvst );
	fprintf(fpw," # nvst\n");
	fprintf(fpl,"nvst = number of wf orbitals\n");

	ipq = 1;
	iloc = 1;
	rcp = 1000.0;
	for( j = 0; j < 4; j++ ){
		if( wf_l[j].orbital >= 0 ){
			for( k = 0; k < v_numorb; k++ ){
				if( wf_l[j].orbital == v_orblist[k] ){
					break;
				}
			}
			if( k >= v_numorb ){
				continue;
			}
			n = nl[wf_l[j].orbital][0];
			l = nl[wf_l[j].orbital][1];
			if( wf_l[j].pptype == 1 ){	/*ultrasoft*/
				iwav = 0;
				/*ipq = 1;*/
			}
			else{
				ipq = 0;
				if( wf_nc_shape == 2 ){		/*T-M*/
					iwav = 0; 
				}
				else if( wf_nc_shape == 3 ){	/*kerker*/
					iwav = 1; 
				}
				else{
					printf("pseudopotential other than TM nor Kerker is input !!!\n");
					fprintf(fpl,"pseudopotential other than TM nor Kerker is input !!!\n");
					/*scanf("%s",deb);*/
					return(-1);
				}
			}
			if( local_val == l ){
				iloc = 0;
			}
			if( rcp > wf_l[j].rcut[0][0] ){
				rcp = wf_l[j].rcut[0][0];
			}
			/*rcut input first is used */
			fprintf(fpw," %4d %4d %12lf %4d %4d %4d", n, l, wf_l[j].rcut[0][0], !(wf_l[j].pptype), iwav, wf_l[j].num_rcut);
			for( i = 0; i < wf_l[j].num_rcut; i++ ){
				fprintf(fpw," %12lf", wf_l[j].rcut[i][1]);
			}
			fprintf(fpw," # nv lv rc ipot iwav nref eref...\n");
		}
	}
	fprintf(fpl,"nv, lv: from wf orbital\n");
	fprintf(fpl,"rc: from wf rcut0 listed first for the orbital\n");
	fprintf(fpl,"ipot = !(pptype)\n");
	fprintf(fpl,"iwav = 0 for US or TM90, 1 for KERKER85\n");
	fprintf(fpl,"nref: number of rcut\n");
	fprintf(fpl,"eref = rcut1\n");

	if( local == 0 ){	/* orbital */
		fprintf(fpw,"    %d                                               ", iloc );
		fprintf(fpw," # iloc\n" );
		fprintf(fpl,"iloc = 0 : outermost orbital\n");
	}
	else if( local == 1 ){ /* gncpp */
		fprintf(fpw,"    0                                               " );
		fprintf(fpw," # iloc\n" );
		fprintf(fpl,"'gncpp' is specified for 'iloc', forced to iloc = 0\n");
	}
	else if( local == 2 ){ /* psv, sbessel */
		iloc = 2;
		fprintf(fpw,"    2                                               " );
		fprintf(fpw," # iloc\n" );
		fprintf(fpl,"iloc = 2 :  for local = psv\n");
	}

	/*assumption: the last orbital specified in wf{} is used.*/
	if( iloc == 0 ){ /*local is included in the valence orbital list*/
		n = nl[wf_l[nvst-1].orbital][0];
		l = nl[wf_l[nvst-1].orbital][1];
		fprintf(fpw," %4d %4d %12lf    0                        ", n, l, rcp );	/*iwavp is always 0*/
		fprintf(fpw," # np lp rcp iwavp\n");
		fprintf(fpl,"np, lp: from orbital listed last for wf\n");
	}
	else if( iloc == 1 ){/*local is NOT included in the valence orbital list*/
		fprintf(fpw," %4d %12lf    0    0.0                     ", local_val, rcp );	/*iwavp is always 0*/
		fprintf(fpw," # lp rcp iwavp erefp\n");
	}
	fprintf(fpl,"rcp : smallest rcut given for wf\n");
	fprintf(fpl,"iwavp = 0 : fixed, not used in the psv program\n");

	fprintf(fpw," %4d                                               ", ipq );	/*nc is specified for all orbitals: 0, else: 1*/
	fprintf(fpw," # ipq\n");
	fprintf(fpl,"ipq = 0 for NC usage, 1 for all US\n");


	/*******************
	l1max = nvst-1;
	l2max = nvst-1;
	for( l1 = 0; l1 <= l1max; l1++ ){
		for( l2 = l1; l2 <= l2max; l2++ ){
			for( i = 0; i < qps_num_rcut; i++ ){
				if( qps_rcut[i][0] == l1 && qps_rcut[i][2] == l2 ){
					break;
				}
			}
			if( i < qps_num_rcut ){
				fprintf(fpw," %12lf", qps_rcut_val[i] );
			}
			else{
				fprintf(fpw," %12lf", 1.5 );
			}
		}
	}
	****************/

	for( i = 0; i < 7; i++ ){
		rin[i] = 0.0;
		rinfg[i] = 0;
	}
	for( i = 0; i < qps_num_rcut; i++ ){
		l3 = qps_rcut[i][4];
		if( qps_rcut_val[i] > rin[l3] ){
			rin[l3] = qps_rcut_val[i];
			rinfg[l3] = 1;
		}
	}
	for( rinmax = 6; rinmax >= 0; rinmax-- ){
		if( rinfg[rinmax] ){
			break;
		}
	}
	for( i = 0; i <= rinmax; i++ ){
		if( rinfg[i] ){
			fprintf(fpw," %12lf", rin[i] );
		}
		else{
			fprintf(fpw," %12lf", 1.0 );
			fprintf(fpl,"rin = 1.0 : fixed for l3 value not given\n");
		}
	}

	/****************************************************/

	fprintf(fpw," # rin...\n" );
	fprintf(fpl,"rin = qps_rcut_val : rin given for different l3\n");

	/*************
	fprintf(fpw," %12lf %12lf %12d             ", wf_gmesh_val1, (wf_gmesh_val2-wf_gmesh_val1)/(wf_gmesh_val0-1),wf_gmesh_val0 );
	fprintf(fpw," # gc gh ng\n");
	fprintf(fpl,"gc = wf_gmesh_val1\n");
	fprintf(fpl,"gh = (wf_gmesh_val2-wf_gmesh_val1)/(wf_gmesh_val0-1)\n");
	fprintf(fpl,"ng = wf_gmesh_val0\n");
	**********/
	fprintf(fpw," %12lf %12lf %12d             ", qps_gmesh_val1, (qps_gmesh_val2-qps_gmesh_val1)/(qps_gmesh_val0-1),qps_gmesh_val0 );
	fprintf(fpw," # gc gh ng\n");
	fprintf(fpl,"gc = qps_gmesh_val1\n");
	fprintf(fpl,"gh = (qps_gmesh_val2-qps_gmesh_val1)/(qps_gmesh_val0-1)\n");
	fprintf(fpl,"ng = qps_gmesh_val0\n");

	fprintf(fpw," %4d                                               ", v_numorb );	/*nsst*/
	fprintf(fpw," # nsst\n");
	fprintf(fpl,"nsst: number of valence electron\n");
	for( i = 0; i < v_numorb; i++ ){
		n = nl[v_orblist[i]][0];
		l = nl[v_orblist[i]][1];
		fprintf(fpw," %4d %4d %12lf                             ", n, l, v_uplist[i] );
		fprintf(fpw," # ns ls swgt\n" );
	}
	fprintf(fpl,"ns, ls: from valence orbital\n");
	fprintf(fpl,"swgt: from valence 'up'\n");
	fprintf(fpw,"END-PSV\n");
	fprintf(fpl,"END-PSV\n");


	fprintf(fpw,"BEGIN-SOLPS\n");
	fprintf(fpl,"BEGIN-SOLPS\n");
	fprintf(fpw,"%s\n",title);
	fprintf(fpw," %4d                                               ", v_numorb );	/*norb*/
	fprintf(fpw," # norb\n");
	fprintf(fpl,"norb: number of valence electron\n");
	for( i = 0; i < v_numorb; i++ ){
		n = nl[v_orblist[i]][0];
		l = nl[v_orblist[i]][1];
		einit = get_init_energy( atom, n, l );
		fprintf(fpw," %4d %4d %12lf %12lf                ", n, l, v_uplist[i], einit /*-((double)atom*atom)/(2.0*n*n)*/ );
		fprintf(fpw," # nv lv occu ein\n");
	}
	fprintf(fpl,"nv, lv: from valence orbital\n");
	fprintf(fpl,"occu: from valence 'up'\n");
	fprintf(fpl,"ein = -(Z^2)/(2n^2) : assumed\n");
	fprintf(fpw,"END-SOLPS\n");
	fprintf(fpl,"END-SOLPS\n");


	fprintf(fpw,"BEGIN-DRV\n");
	fprintf(fpl,"BEGIN-DRV\n");
	fprintf(fpw,"%s\n",title);
	fprintf(fpw," %12lf %12lf %12d             ", ld_emesh_val1,ld_emesh_val2,ld_emesh_val0);
	fprintf(fpw," # emin emax neng\n");
	fprintf(fpl,"emin = ld_emesh_val1\n");
	fprintf(fpl,"emax = ld_emesh_val2\n");
	fprintf(fpl,"neng = ld_emesh_val0\n");
	fprintf(fpw," %12lf                                       ", ld_rcut);
	fprintf(fpw," # rdrv\n");
	fprintf(fpl,"rdrv = ld_rcut\n");
	fprintf(fpw,"END-DRV\n");
	fprintf(fpl,"END-DRV\n");
	fclose(fpw);

	fprintf(fpl,"}\n");
	return( 0 );
}

/********************************************************************/
int getfiletype_name( char *instr, int *mode, char *fname )
{
int i, len;
char extbuf[16];

	len = strlen( instr );
	for( i = 0; i < len; i++ ){
		if( instr[i] == ':' ){
			instr[i] = 0;
			break;
		}
	}
	if( i < len ){
		strlower( instr );
		if( strcmp( instr, "ciao" ) == 0 ){
			*mode = 0;
		}
		else if( strcmp( instr, "gncpp" ) == 0 ){
			*mode = 1;
		}
		else if( strcmp( instr, "psv" ) == 0 ){
			*mode = 2;
		}
		else if( strcmp( instr, "psvpp" ) == 0 ){
			*mode = 2;
		}
		else{
			printf("unknown file type specified !!!\n");
			return( -1 );
		}
		strcpy( fname, instr+i+1 );
	}
	else{
		for( i = len-1; i >= 0; i-- ){
			if( instr[i] == '.' ){
				break;
			}
		}
		if( i < 0 ){
			extbuf[0] = 0;
		}
		else{
			strcpy( extbuf, instr+i+1 );
			strlower( extbuf );
		}

		if( strcmp( extbuf, "gncpp" ) == 0 ){
			*mode = 1;
		}
		else if( strcmp( extbuf, "psv" ) == 0 ){
			*mode = 2;
		}
		else if( strcmp( extbuf, "psvpp" ) == 0 ){
			*mode = 2;
		}
		else{
			*mode = 0;
		}
		strcpy( fname, instr );
	}
	return( *mode );
}


int main( int argc, char **argv )
{
int mode, rmode, wmode, len;
char argbuf[128], rmodename[16], wmodename[16];
char deb[16];
time_t ltime;

	if( argc == 2 ){
		strcpy( argbuf, argv[1] );
		strlower( argbuf );
		if( strcmp( argbuf, "-i" ) == 0 || strcmp( argbuf, "--interactive" ) == 0){
			printf("ciao  -> ciao  : 0\n");
			printf("ciao  -> gncpp : 1\n");
			printf("ciao  -> psvpp : 2\n");
			printf("gncpp -> ciao  : 3\n");
			printf("gncpp -> gncpp : 4\n");
			printf("gncpp -> psvpp : 5\n");
			printf("psvpp -> ciao  : 6\n");
			printf("psvpp -> gncpp : 7\n");
			printf("psvpp -> psvpp : 8\n");
			scanf("%d",&mode);

			while( mode < 0 || mode > 8 ){
				printf("Please enter 0-8 !!!\n");
				scanf("%d", &mode);
			}
			rmode = mode/3;
			wmode = mode % 3;

			printf("input file name=");
			scanf("%s",rfile);
			printf("output file name=");
			scanf("%s",wfile);
		}
		else if( strcmp( argbuf, "-h" ) == 0 || strcmp( argbuf, "--help" ) == 0){
			printf("ppconv [{ciao,gncpp,psv}:]input-filename [{ciao,gncpp,psv}:]output-filename\n");
			printf("-i, --interactive        interactive mode\n");
			printf("-h, --help               help\n");
			exit(0);
		}
		else{
			printf("ppconv [{ciao,gncpp,psv}:]input-filename [{ciao,gncpp,psv}:]output-filename\n");
			printf("-i, --interactive        interactive mode\n");
			printf("-h, --help               help\n");
			exit(0);
		}
	}
	else if( argc == 3 ){
		strcpy( argbuf, argv[1] );
		if( getfiletype_name( argbuf, &rmode, rfile ) < 0 ){
			exit( 1 );
		}
		strcpy( argbuf, argv[2] );
		if( getfiletype_name( argbuf, &wmode, wfile ) < 0 ){
			exit( 1 );
		}
	}
	else{
		printf("wrong number of arguments !!!\n");
		printf("ppconv [{ciao,gncpp,psv}:]input-filename [{ciao,gncpp,psv}:]output-filename\n");
		printf("-i, --interactive        interactive mode\n");
		printf("-h, --help               help\n");
		exit(0);
	}

	/*****************************************************************
	open log-file and write date and input- and output file names
	*****************************************************************/
	if( (fpl=fopen( "ppconv.log", "a" )) == NULL ){
		printf("Can't open log-file !!!\n");
		exit(1);
	}
	switch(rmode){
		case 0: strcpy( rmodename, "CIAO" ); break;
		case 1: strcpy( rmodename, "GNCPP" ); break;
		case 2: strcpy( rmodename, "PSVPP" ); break;
	}
	switch(wmode){
		case 0: strcpy( wmodename, "CIAO" ); break;
		case 1: strcpy( wmodename, "GNCPP" ); break;
		case 2: strcpy( wmodename, "PSVPP" ); break;
	}
	time( &ltime );

	fprintf(fpl,"\n");
    fprintf(fpl, "%s", ctime( &ltime ) );
	fprintf(fpl,"%s:%s ---> %s:%s\n\n",rmodename,rfile,wmodename,wfile);

	/*****************************************************************
	reading input file
	*****************************************************************/
	if( rmode == 0 ){		/*file type: ATOM*/
		if( read_std_form( rfile ) < 0 ){
			printf("read-file error !!!\n");
			fprintf(fpl,"read-file error !!!\n");
			goto ERROREXIT;
		}
	}
	else if( rmode == 1 ){	/*file type: GNCPP*/
		if( read_gncpp_form( rfile ) < 0 ){
			printf("read-file error !!!\n");
			fprintf(fpl,"read-file error !!!\n");
			goto ERROREXIT;
		}
	}
	else{					/*file type: PSV*/
		if( read_psv_form( rfile ) < 0 ){
			printf("read-file error !!!\n");
			fprintf(fpl,"read-file error !!!\n");
			goto ERROREXIT;
		}
	}

	/*****************************************************************
	writing output file
	*****************************************************************/
	if( wmode == 0 ){		/*file type: ATOM*/
		if( write_std_form( wfile ) < 0 ){
			goto ERROREXIT;
		}
	}
	else if( wmode == 1 ){	/*file type: GNCPP*/
		if( write_gncpp_form( wfile ) < 0 ){
			goto ERROREXIT;
		}
	}
	else{					/*file type: PSV*/
		if( write_psv_form( wfile ) < 0 ){
			goto ERROREXIT;
		}
	}

	fprintf(fpl,"conversion terminated normally\n");
	fprintf(fpl,"----------------------------------------------------------------\n");
	fclose( fpl );
	exit(0);

ERROREXIT:
	fprintf(fpl,"conversion terminated by error !!!\n");
	fprintf(fpl,"----------------------------------------------------------------\n");
	fclose( fpl );
	exit(1);

}
