!
!  Software name : CIAO (Code for Investigating Atomic Orbitals)
!  Subroutine(s) : opt_sbessel_fn, calc_sbessel_fn, bessel_js
!  Function(s)   : fn_inv_xcot, fn_js
!  Author(s)     : Masakuni Okamoto (August 25, 2003)
!
!
!  Contact address :  Phase System Consortium
!                     E-mail: phase_system@nims.go.jp URL https://azuma.nims.go.jp
!
!
!  The license of the code and contact address :
!  See the files, COPYRIGHT and LICENSE (or LICENSE_J.pdf)
!
!=====================================================================
   subroutine opt_sbessel_fn(ier,ifile, &
                             nrc,rc,fc,dfc,a,b)
!=====================================================================
!
!  This subroutine calculates partial core charge density
!  by Louie et al.
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   integer,intent(in)  :: ifile, nrc
   real(8),intent(in)  :: rc, fc, dfc
   integer,intent(out) :: ier
   real(8),intent(out) :: a, b
   real(8) :: fn_inv_xcot
   ier = 0
   b = fn_inv_xcot(1.d0 + rc*dfc/fc) / rc
   a = rc*fc / sin(b*rc)
   end subroutine opt_sbessel_fn
   
!=====================================================================
   subroutine calc_sbessel_fn(n1,n2,rpos,a,b,fps)
!=====================================================================
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   integer,intent(in)  :: n1, n2
   real(8),intent(in)  :: rpos(*), a, b
   real(8),intent(out) :: fps(*)
   integer :: ir
   real(8) :: r
   do ir = n1,n2
      r = rpos(ir)
      fps(ir) = a * sin(b*r) / r
   end do
   end subroutine calc_sbessel_fn

!=====================================================================
   function fn_inv_xcot(x)
!=====================================================================
!
!  This function calculates the inverse function of x*cot(x)
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   real(8) :: fn_inv_xcot
   real(8),intent(in) :: x
   real(8) :: xmin, xmid, xmax, ymin, ymid, ymax, diff
   real(8),parameter :: &
      PI = 3.1415926535897932384626433832795028d0, &
      EPS = 1.d-18
   if (x > 1.d0) then
      fn_inv_xcot = -1.d0 ; go to 99
   end if
   ymin = 0.5d0 * PI
   xmin = ymin / tan(ymin)
   do while (xmin < x)
      ymin = 0.5d0 * ymin
      xmin = ymin / tan(ymin)
   end do
   ymax = 0.5d0 * (ymin + PI)
   xmax = ymax / tan(ymax)
   do while (xmax > x)
      ymax = 0.5d0 * (ymax + PI)
      xmax = ymax / tan(ymax)
   end do
   ymid = 0.5d0 * (ymin + ymax)
   diff = ymax - ymin
   do while (diff > EPS)
      xmin = ymin / tan(ymin)
      xmid = ymid / tan(ymid)
      xmax = ymax / tan(ymax)
      if (x > xmid) then
         ymax = ymid
      else
         ymin = ymid
      end if
      ymid = 0.5d0 * (ymin + ymax)
      diff = ymax - ymin
   end do
   fn_inv_xcot = ymid
99 continue
   end function fn_inv_xcot

!=====================================================================
   subroutine bessel_js(nj,np,xp,js)
!=====================================================================
!
!  The Spherical Bessel function : j(n,x)
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   integer,intent(in)  :: nj, np
   real(8),intent(in)  :: xp(1:np)
   real(8),intent(out) :: js(1:np)
   integer :: ip, n
   real(8) :: s, c, x, x2
   real(8),parameter :: EPS = 1.0d0
   real(8),parameter :: &
      cc10 =  1.0000000000000000000d0,   &
      cc20 = -1.6666666666666666667d-01, &
      cc30 =  8.3333333333333333333d-03, &
      cc40 = -1.9841269841269841270d-04, &
      cc50 =  2.7557319223985890653d-06, &
      cc60 = -2.5052108385441718775d-08, &
      cc70 =  1.6059043836821614599d-10, &
      cc80 = -7.6471637318198164759d-13, &
      cc11 =  3.3333333333333333333d-01, &
      cc21 = -3.3333333333333333333d-02, &
      cc31 =  1.1904761904761904762d-03, &
      cc41 = -2.2045855379188712522d-05, &
      cc51 =  2.5052108385441718775d-07, &
      cc61 = -1.9270852604185937519d-09, &
      cc71 =  1.0706029224547743066d-11, &
      cc81 = -4.4983316069528332211d-14, &
      cc12 =  6.6666666666666666667d-02, &
      cc22 = -4.7619047619047619048d-03, &
      cc32 =  1.3227513227513227513d-04, &
      cc42 = -2.0041686708353375020d-06, &
      cc52 =  1.9270852604185937519d-08, &
      cc62 = -1.2847235069457291680d-10, &
      cc72 =  6.2976642497339665096d-13, &
      cc82 = -2.3675429510278069585d-15, &
      cc13 =  9.5238095238095238095d-03, &
      cc23 = -5.2910052910052910053d-04, &
      cc33 =  1.2025012025012025012d-05, &
      cc43 = -1.5416682083348750015d-07, &
      cc53 =  1.2847235069457291680d-09, &
      cc63 = -7.5571970996807598115d-12, &
      cc73 =  3.3145601314389297419d-14, &
      cc83 = -1.1274014052513366469d-16, &
      cc14 =  1.0582010582010582011d-03, &
      cc24 = -4.8100048100048100048d-05, &
      cc34 =  9.2500092500092500093d-07, &
      cc44 = -1.0277788055565833344d-08, &
      cc54 =  7.5571970996807598115d-11, &
      cc64 = -3.9774721577267156903d-13, &
      cc74 =  1.5783619673518713057d-15, &
      cc84 = -4.9017452402232028126d-18, &
      cc15 =  9.6200096200096200096d-05, &
      cc25 = -3.7000037000037000037d-06, &
      cc35 =  6.1666728333395000062d-08, &
      cc45 = -6.0457576797446078492d-10, &
      cc55 =  3.9774721577267156903d-12, &
      cc65 = -1.8940343608222455668d-14, &
      cc75 =  6.8624433363124839376d-17, &
      cc85 = -1.9606980960892811250d-19, &
      cc16 =  7.4000074000074000074d-06, &
      cc26 = -2.4666691333358000025d-07, &
      cc36 =  3.6274546078467647095d-09, &
      cc46 = -3.1819777261813725522d-11, &
      cc56 =  1.8940343608222455668d-13, &
      cc66 = -8.2349320035749807252d-16, &
      cc76 =  2.7449773345249935751d-18, &
      cc86 = -7.2618448003306708335d-21, &
      cc17 =  4.9333382666716000049d-07, &
      cc27 = -1.4509818431387058838d-08, &
      cc37 =  1.9091866357088235313d-10, &
      cc47 = -1.5152274886577964534d-12, &
      cc57 =  8.2349320035749807252d-15, &
      cc67 = -3.2939728014299922901d-17, &
      cc77 =  1.0166582720462939167d-19, &
      cc87 = -2.5040844139071278736d-22, &
      cc18 =  2.9019636862774117676d-08, &
      cc28 = -7.6367465428352941253d-10, &
      cc38 =  9.0913649319467787206d-12, &
      cc48 = -6.5879456028599845801d-14, &
      cc58 =  3.2939728014299922901d-16, &
      cc68 = -1.2199899264555527000d-18, &
      cc78 =  3.5057181794699790231d-21, &
      cc88 = -8.0776916577649286246d-24   
   n = abs(nj)
   select case (n)
   case (0)
      do ip = 1,np
         x = xp(ip)
         if (abs(x) < EPS) then
            x2 = x*x
            js(ip) = (cc10+x2*(cc20+x2*(cc30+x2*(cc40 &
                    +x2*(cc50+x2*(cc60+x2*(cc70+x2*cc80 &
                     )))))))
         else
            s = sin(x)
            js(ip) = s/x
         end if
      end do
   case (1)
      do ip = 1,np
         x = xp(ip) ; x2 = x*x
         if (abs(x) < EPS) then
            js(ip) = (cc11+x2*(cc21+x2*(cc31+x2*(cc41 &
                    +x2*(cc51+x2*(cc61+x2*(cc71+x2*cc81 &
                     )))))))*x
         else
            s = sin(x) ; c = cos(x)
            js(ip) = (s-x*c)/x2
         end if
      end do
   case (2)
      do ip = 1,np
         x = xp(ip) ; x2 = x*x
         if (abs(x) < EPS) then
            js(ip) = (cc12+x2*(cc22+x2*(cc32+x2*(cc42 &
                    +x2*(cc52+x2*(cc62+x2*(cc72+x2*cc82 &
                     )))))))*x2
         else
            s = sin(x) ; c = cos(x)
            js(ip) = ((3.d0-x2)*s-3.d0*x*c)/(x2*x)
         end if
      end do
   case (3)
      do ip = 1,np
         x = xp(ip) ; x2 = x*x
         if (abs(x) < EPS) then
            js(ip) = (cc13+x2*(cc23+x2*(cc33+x2*(cc43 &
                    +x2*(cc53+x2*(cc63+x2*(cc73+x2*cc83 &
                     )))))))*x2*x
         else
            s = sin(x) ; c = cos(x)
            js(ip) = ((15.d0-6.d0*x2)*s-(15.d0-x2)*x*c)/(x2*x2)
         end if
      end do
   case (4)
      do ip = 1,np
         x = xp(ip) ; x2 = x*x
         if (abs(x) < EPS) then
            js(ip) = (cc14+x2*(cc24+x2*(cc34+x2*(cc44 &
                +x2*(cc54+x2*(cc64+x2*(cc74+x2*cc84 &
                 )))))))*x2*x2
         else
            s = sin(x) ; c = cos(x)
            js(ip) = ((105.d0+(-45.d0+x2)*x2)*s &
                    -(105.d0-10.d0*x2)*x*c)/(x2*x2*x)
         end if
      end do
   case (5)
      do ip = 1,np
         x = xp(ip) ; x2 = x*x
         if (abs(x) < EPS) then
            js(ip) = (cc15+x2*(cc25+x2*(cc35+x2*(cc45 &
                    +x2*(cc55+x2*(cc65+x2*(cc75+x2*cc85 &
                     )))))))*x2*x2*x
         else
            s = sin(x) ; c = cos(x)
            js(ip) = ((945.d0+(-420.d0+15.d0*x2)*x2)*s &
                    -(945.d0+(-105.d0+x2)*x2)*x*c)/(x2*x2*x2)
         end if
      end do
   case (6)
      do ip = 1,np
         x = xp(ip) ; x2 = x*x
         if (abs(x) < EPS) then
            js(ip) = (cc16+x2*(cc26+x2*(cc36+x2*(cc46 &
                    +x2*(cc56+x2*(cc66+x2*(cc76+x2*cc86 &
                     )))))))*x2*x2*x2
         else
            s = sin(x) ; c = cos(x)
            js(ip) = ((10395.d0+(-4725.d0+(210.d0-x2)*x2)*x2)*s &
                    -(10395.d0+(-1260.d0+21.d0*x2)*x2)*x*c &
                     )/(x2*x2*x2*x)
         end if
      end do
   case (7)
      do ip = 1,np
         x = xp(ip) ; x2 = x*x
         if (abs(x) < EPS) then
            js(ip) = (cc17+x2*(cc27+x2*(cc37+x2*(cc47 &
                    +x2*(cc57+x2*(cc67+x2*(cc77+x2*cc87 &
                     )))))))*x2*x2*x2*x
         else
            s = sin(x) ; c = cos(x)
            js(ip) = &
               ((135135.d0+(-62370.d0+(3150.d0-28.d0*x2)*x2)*x2)*s &
                -(135135.d0+(-17325.d0+(378.d0-x2)*x2)*x2)*x*c &
               )/(x2*x2*x2*x2)
         end if
      end do
   case (8)
      do ip = 1,np
         x = xp(ip) ; x2 = x*x
         if (abs(x) < EPS) then
            js(ip) = (cc18+x2*(cc28+x2*(cc38+x2*(cc48 &
                    +x2*(cc58+x2*(cc68+x2*(cc78+x2*cc88 &
                     )))))))*x2*x2*x2*x2
         else
            s = sin(x) ; c = cos(x)
            js(ip) = ( &
      (2027025.d0+(-945945.d0+(51975.d0+(-630.d0+x2)*x2)*x2)*x2)*s &
      - (2027025.d0+(-270270.d0+(6930.d0-36.d0*x2)*x2)*x2)*x*c &
                     )/(x2*x2*x2*x2*x)
         end if
      end do
   case default
      js(:) = 0.d0
   end select
   end subroutine bessel_js

!=====================================================================
   function fn_js(n,x) 
!=====================================================================
!
!  The Spherical Bessel function : j(n,x)
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   integer,intent(in) :: n
   real(8),intent(in) :: x
   real(8)            :: fn_js
   real(8) :: cc(8,0:8), s, c, x2
   real(8),parameter :: EPS = 1.0d0
   x2 = x*x ; s = sin(x) ; c = cos(x)
   cc(:,:) = 0.d0
   cc(1,0) =  1.0000000000000000000d0  
   cc(2,0) = -1.6666666666666666667d-01
   cc(3,0) =  8.3333333333333333333d-03
   cc(4,0) = -1.9841269841269841270d-04
   cc(5,0) =  2.7557319223985890653d-06
   cc(6,0) = -2.5052108385441718775d-08
   cc(7,0) =  1.6059043836821614599d-10
   cc(8,0) = -7.6471637318198164759d-13
   cc(1,1) =  3.3333333333333333333d-01
   cc(2,1) = -3.3333333333333333333d-02
   cc(3,1) =  1.1904761904761904762d-03
   cc(4,1) = -2.2045855379188712522d-05
   cc(5,1) =  2.5052108385441718775d-07
   cc(6,1) = -1.9270852604185937519d-09
   cc(7,1) =  1.0706029224547743066d-11
   cc(8,1) = -4.4983316069528332211d-14
   cc(1,2) =  6.6666666666666666667d-02
   cc(2,2) = -4.7619047619047619048d-03
   cc(3,2) =  1.3227513227513227513d-04
   cc(4,2) = -2.0041686708353375020d-06
   cc(5,2) =  1.9270852604185937519d-08
   cc(6,2) = -1.2847235069457291680d-10
   cc(7,2) =  6.2976642497339665096d-13
   cc(8,2) = -2.3675429510278069585d-15
   cc(1,3) =  9.5238095238095238095d-03
   cc(2,3) = -5.2910052910052910053d-04
   cc(3,3) =  1.2025012025012025012d-05
   cc(4,3) = -1.5416682083348750015d-07
   cc(5,3) =  1.2847235069457291680d-09
   cc(6,3) = -7.5571970996807598115d-12
   cc(7,3) =  3.3145601314389297419d-14
   cc(8,3) = -1.1274014052513366469d-16
   cc(1,4) =  1.0582010582010582011d-03
   cc(2,4) = -4.8100048100048100048d-05
   cc(3,4) =  9.2500092500092500093d-07
   cc(4,4) = -1.0277788055565833344d-08
   cc(5,4) =  7.5571970996807598115d-11
   cc(6,4) = -3.9774721577267156903d-13
   cc(7,4) =  1.5783619673518713057d-15
   cc(8,4) = -4.9017452402232028126d-18
   cc(1,5) =  9.6200096200096200096d-05
   cc(2,5) = -3.7000037000037000037d-06
   cc(3,5) =  6.1666728333395000062d-08
   cc(4,5) = -6.0457576797446078492d-10
   cc(5,5) =  3.9774721577267156903d-12
   cc(6,5) = -1.8940343608222455668d-14
   cc(7,5) =  6.8624433363124839376d-17
   cc(8,5) = -1.9606980960892811250d-19
   cc(1,6) =  7.4000074000074000074d-06
   cc(2,6) = -2.4666691333358000025d-07
   cc(3,6) =  3.6274546078467647095d-09
   cc(4,6) = -3.1819777261813725522d-11
   cc(5,6) =  1.8940343608222455668d-13
   cc(6,6) = -8.2349320035749807252d-16
   cc(7,6) =  2.7449773345249935751d-18
   cc(8,6) = -7.2618448003306708335d-21
   cc(1,7) =  4.9333382666716000049d-07
   cc(2,7) = -1.4509818431387058838d-08
   cc(3,7) =  1.9091866357088235313d-10
   cc(4,7) = -1.5152274886577964534d-12
   cc(5,7) =  8.2349320035749807252d-15
   cc(6,7) = -3.2939728014299922901d-17
   cc(7,7) =  1.0166582720462939167d-19
   cc(8,7) = -2.5040844139071278736d-22
   cc(1,8) =  2.9019636862774117676d-08
   cc(2,8) = -7.6367465428352941253d-10
   cc(3,8) =  9.0913649319467787206d-12
   cc(4,8) = -6.5879456028599845801d-14
   cc(5,8) =  3.2939728014299922901d-16
   cc(6,8) = -1.2199899264555527000d-18
   cc(7,8) =  3.5057181794699790231d-21
   cc(8,8) = -8.0776916577649286246d-24
   if (abs(x) < EPS) then
      if (abs(n) <= 8) then
         fn_js = (cc(1,n)+x2*(cc(2,n)+x2*(cc(3,n)+x2*(cc(4,n) &
                +x2*(cc(5,n)+x2*(cc(6,n)+x2*(cc(7,n)+x2*cc(8,n) &
                 )))))))*x**n
      else
         fn_js = 0.d0
      end if
   else
      select case (abs(n))
      case (0)
         fn_js = s/x
      case (1)
         fn_js = (s-x*c)/x2
      case (2)
         fn_js = ((3.d0-x2)*s-3.d0*x*c)/x/x2
      case (3)
         fn_js = ((15.d0-6.d0*x2)*s &
                  -(15.d0-x2)*x*c)/x2/x2
      case (4)
         fn_js = ((105.d0+(-45.d0+x2)*x2)*s &
                  -(105.d0-10.d0*x2)*x*c)/x/x2/x2
      case (5)
         fn_js = ((945.d0+(-420.d0+15.d0*x2)*x2)*s &
                  -(945.d0+(-105.d0+x2)*x2)*x*c)/x2/x2/x2
      case (6)
         fn_js = ((10395.d0+(-4725.d0+(210.d0-x2)*x2)*x2)*s &
                  -(10395.d0+(-1260.d0+21.d0*x2)*x2)*x*c &
                 )/x/x2/x2/x2
      case (7)
         fn_js = ((135135.d0+(-62370.d0+(3150.d0-28.d0*x2)*x2)*x2)*s &
                  -(135135.d0+(-17325.d0+(378.d0-x2)*x2)*x2)*x*c &
                 )/x2/x2/x2/x2
      case (8)
         fn_js = ( &
         (2027025.d0+(-945945.d0+(51975.d0+(-630.d0+x2)*x2)*x2)*x2)*s &
         - (2027025.d0+(-270270.d0+(6930.d0-36.d0*x2)*x2)*x2)*x*c &
                 )/x/x2/x2/x2/x2
      case default
         fn_js = 0.d0
      end select
   end if
   end function fn_js
