! ************************************************************* 
!
!   This is a software package CIAO
!
!     developed as a part of the national project "Research and 
!     Development of Innovative Simulation software",which is   
!     supported by the next-generation IT program of MEXT of Japan
!
!   Version history: 
!
!     4.0:  2013/03/01
!           codes for spin-polarized pseudopotential generation are added
!     4.1:  2013/11/22 - 
!           Info of core wfns and energy contributions can be added to gncpp2
!     4.2:  2014/07/23 - 
!           gncpp2 can be geregated even when nmesh /= 1501
!
! ************************************************************* 
!
!  Software name : CIAO (Code for Investigating Atomic Orbitals)
!  Subroutine(s) : pc_adams_4, pc_adams_5, pc_adams_6, pc_adams_7
!                : pc_adams_8, pc_adams_4ps, pc_adams_5ps
!                : pc_adams_6ps, pc_adams_7ps, pc_adams_8ps
!  Author(s)     : Masakuni Okamoto (August 25, 2003)
!
!
!  Contact address :  Phase System Consortium
!                     E-mail: phase_system@nims.go.jp URL https://azuma.nims.go.jp
!
!
!  The license of the code and contact address :
!  See the files, COPYRIGHT and LICENSE (or LICENSE_J.pdf)
!
!=====================================================================
   subroutine pc_adams_4(ier,ir,id,ll,kk,ee,nec,ispin)
!===================================================================== 
!
!  Predicts & Corrects chi_g(ir+id) & chi_f(ir+id)
!  using values at xpos(ir), xpos(ir-id), xpos(ir-2*id)
!  by the Adams Predictor-Corrector Method (3rd order)
!     
!  M. Okamoto
!
!---------------------------------------------------------------------
   use parameters   
   implicit none
   integer,intent(in)    :: ir, ll, kk, nec, ispin
   real(8),intent(in)    :: ee
   integer,intent(inout) :: id
   integer,intent(out)   :: ier
   integer :: iec, ll_core
   real(8) :: dx12, &
              g0, g1, g2, dg0, dg1, dg2, g9, dg9, &
              f0, f1, f2, df0, df1, df2, f9, df9
   ier = 0
   if (is_core == PATOM) then
      ll_core = ll
   else
      ll_core = lmax_core
   end if
   id   = sign(1, id)
   dx12 = dx / 12.d0 * dble(id)
   select case (id)
   case(+1)
      g0  =   chi_gl(ir     ) ; f0  =   chi_fl(ir     )
      g1  =   chi_gl(ir-1*id) ; f1  =   chi_fl(ir-1*id)
      g2  =   chi_gl(ir-2*id) ; f2  =   chi_fl(ir-2*id)
      dg0 = dxchi_gl(ir     ) ; df0 = dxchi_fl(ir     )
      dg1 = dxchi_gl(ir-1*id) ; df1 = dxchi_fl(ir-1*id)
      dg2 = dxchi_gl(ir-2*id) ; df2 = dxchi_fl(ir-2*id)
   case(-1)
      g0  =   chi_gr(ir     ) ; f0  =   chi_fr(ir     )
      g1  =   chi_gr(ir-1*id) ; f1  =   chi_fr(ir-1*id)
      g2  =   chi_gr(ir-2*id) ; f2  =   chi_fr(ir-2*id)
      dg0 = dxchi_gr(ir     ) ; df0 = dxchi_fr(ir     )
      dg1 = dxchi_gr(ir-1*id) ; df1 = dxchi_fr(ir-1*id)
      dg2 = dxchi_gr(ir-2*id) ; df2 = dxchi_fr(ir-2*id)
   end select
   g9 = g0 + dx12 * (23.d0*dg0 - 16.d0*dg1 + 5.d0*dg2)
   f9 = f0 + dx12 * (23.d0*df0 - 16.d0*df1 + 5.d0*df2)
   do iec = 1,nec
      select case (is_calc)
      case (NONREL)
         call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff(ir+id,ispin,ll_core))
      case (SREL)
         call dgdx_dfdx_srel(rpos(ir+id),g9,f9,dg9,df9, &
                             ll,ee,veff(ir+id,ispin,ll_core))
      case (REL)
         call dgdx_dfdx_rel(rpos(ir+id),g9,f9,dg9,df9, &
                            kk,ee,veff(ir+id,ispin,ll_core))
      end select
      g9 = g0 + dx12 * (5.d0*dg9 + 8.d0*dg0 - dg1)
      f9 = f0 + dx12 * (5.d0*df9 + 8.d0*df0 - df1)
   end do 
   select case (is_calc)
      case (NONREL)
         call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff(ir+id,ispin,ll_core))
      case (SREL)
         call dgdx_dfdx_srel(rpos(ir+id),g9,f9,dg9,df9, &
                             ll,ee,veff(ir+id,ispin,ll_core))
      case (REL)
         call dgdx_dfdx_rel(rpos(ir+id),g9,f9,dg9,df9, &
                            kk,ee,veff(ir+id,ispin,ll_core))
   end select
   select case (id)
   case(+1)
        chi_gl(ir+id) =  g9 ;   chi_fl(ir+id) =  f9
      dxchi_gl(ir+id) = dg9 ; dxchi_fl(ir+id) = df9
   case(-1)
        chi_gr(ir+id) =  g9 ;   chi_fr(ir+id) =  f9
      dxchi_gr(ir+id) = dg9 ; dxchi_fr(ir+id) = df9
   end select
   end subroutine pc_adams_4

!=====================================================================
   subroutine pc_adams_5(ier,ir,id,ll,kk,ee,nec,ispin)
!=====================================================================
!
!  Predicts & Corrects chi_g(ir+id) & chi_f(ir+id)
!  using values at xpos(ir), xpos(ir-id), xpos(ir-2*id), xpos(ir-3*id)
!  by the Adams Predictor-Corrector Method
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   use parameters   
   implicit none
   integer,intent(in)    :: ir, ll, kk, nec, ispin
   real(8),intent(in)    :: ee
   integer,intent(inout) :: id
   integer,intent(out)   :: ier
   integer :: iec, ll_core
   real(8) :: dx24, &
              g0, g1, g2, g3, dg0, dg1, dg2, dg3, g9, dg9, &
              f0, f1, f2, f3, df0, df1, df2, df3, f9, df9
   ier = 0
   if (is_core == PATOM) then
      ll_core = ll
   else
      ll_core = lmax_core
   end if
   id   = sign(1, id)
   dx24 = dx / 24.d0 * dble(id)
   select case (id)
   case(+1)
      g0  =   chi_gl(ir     ) ; f0  =   chi_fl(ir     )
      g1  =   chi_gl(ir-1*id) ; f1  =   chi_fl(ir-1*id)
      g2  =   chi_gl(ir-2*id) ; f2  =   chi_fl(ir-2*id)
      g3  =   chi_gl(ir-3*id) ; f3  =   chi_fl(ir-3*id)
      dg0 = dxchi_gl(ir     ) ; df0 = dxchi_fl(ir     )
      dg1 = dxchi_gl(ir-1*id) ; df1 = dxchi_fl(ir-1*id)
      dg2 = dxchi_gl(ir-2*id) ; df2 = dxchi_fl(ir-2*id)
      dg3 = dxchi_gl(ir-3*id) ; df3 = dxchi_fl(ir-3*id)
   case(-1)
      g0  =   chi_gr(ir     ) ; f0  =   chi_fr(ir     )
      g1  =   chi_gr(ir-1*id) ; f1  =   chi_fr(ir-1*id)
      g2  =   chi_gr(ir-2*id) ; f2  =   chi_fr(ir-2*id)
      g3  =   chi_gr(ir-3*id) ; f3  =   chi_fr(ir-3*id)
      dg0 = dxchi_gr(ir     ) ; df0 = dxchi_fr(ir     )
      dg1 = dxchi_gr(ir-1*id) ; df1 = dxchi_fr(ir-1*id)
      dg2 = dxchi_gr(ir-2*id) ; df2 = dxchi_fr(ir-2*id)
      dg3 = dxchi_gr(ir-3*id) ; df3 = dxchi_fr(ir-3*id)
   end select
   g9 = g0 + dx24 * (55.d0*dg0 - 59.d0*dg1 + 37.d0*dg2 - 9.d0*dg3)
   f9 = f0 + dx24 * (55.d0*df0 - 59.d0*df1 + 37.d0*df2 - 9.d0*df3)
   do iec = 1,nec 
      select case (is_calc)
      case (NONREL)
         call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff(ir+id,ispin,ll_core))
      case (SREL)
         call dgdx_dfdx_srel(rpos(ir+id),g9,f9,dg9,df9, &
                             ll,ee,veff(ir+id,ispin,ll_core))
      case (REL)
         call dgdx_dfdx_rel(rpos(ir+id),g9,f9,dg9,df9, &
                            kk,ee,veff(ir+id,ispin,ll_core))
      end select
      g9 = g0 + dx24 * (9.d0*dg9 + 19.d0*dg0 - 5.d0*dg1 + dg2)
      f9 = f0 + dx24 * (9.d0*df9 + 19.d0*df0 - 5.d0*df1 + df2)
   end do   
   select case (is_calc)
   case (NONREL)
      call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                            ll,ee,veff(ir+id,ispin,ll_core))
   case (SREL)
      call dgdx_dfdx_srel(rpos(ir+id),g9,f9,dg9,df9, &
                          ll,ee,veff(ir+id,ispin,ll_core))
   case (REL)
      call dgdx_dfdx_rel(rpos(ir+id),g9,f9,dg9,df9, &
                         kk,ee,veff(ir+id,ispin,ll_core))
   end select
   select case (id)
   case(+1)
        chi_gl(ir+id) =  g9 ;   chi_fl(ir+id) =  f9
      dxchi_gl(ir+id) = dg9 ; dxchi_fl(ir+id) = df9
   case(-1)
        chi_gr(ir+id) =  g9 ;   chi_fr(ir+id) =  f9
      dxchi_gr(ir+id) = dg9 ; dxchi_fr(ir+id) = df9
   end select
   end subroutine pc_adams_5

!=====================================================================
   subroutine pc_adams_6(ier,ir,id,ll,kk,ee,nec,ispin)
!=====================================================================
!
!  Predicts & Corrects chi_g(ir+id) & chi_f(ir+id)
!  using values at xpos(ir), xpos(ir-id), xpos(ir-2*id), xpos(ir-3*id)
!  by the Adams Predictor-Corrector Method
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   use parameters   
   implicit none
   integer,intent(in)    :: ir, ll, kk, nec, ispin
   real(8),intent(in)    :: ee
   integer,intent(inout) :: id
   integer,intent(out)   :: ier
   integer :: iec, ll_core
   real(8) :: dx24, dx720, &
              g0, g1, g2, g3, dg0, dg1, dg2, dg3, g9, dg9, &
              f0, f1, f2, f3, df0, df1, df2, df3, f9, df9
   ier = 0
   if (is_core == PATOM) then
      ll_core = ll
   else
      ll_core = lmax_core
   end if
   id    = sign(1, id)
   dx24  = dx /  24.d0 * dble(id)
   dx720 = dx / 720.d0 * dble(id)
   select case (id)
   case(+1)
      g0  =   chi_gl(ir     ) ; f0  =   chi_fl(ir     )
      g1  =   chi_gl(ir-1*id) ; f1  =   chi_fl(ir-1*id)
      g2  =   chi_gl(ir-2*id) ; f2  =   chi_fl(ir-2*id)
      g3  =   chi_gl(ir-3*id) ; f3  =   chi_fl(ir-3*id)
      dg0 = dxchi_gl(ir     ) ; df0 = dxchi_fl(ir     )
      dg1 = dxchi_gl(ir-1*id) ; df1 = dxchi_fl(ir-1*id)
      dg2 = dxchi_gl(ir-2*id) ; df2 = dxchi_fl(ir-2*id)
      dg3 = dxchi_gl(ir-3*id) ; df3 = dxchi_fl(ir-3*id)
   case(-1)
      g0  =   chi_gr(ir     ) ; f0  =   chi_fr(ir     )
      g1  =   chi_gr(ir-1*id) ; f1  =   chi_fr(ir-1*id)
      g2  =   chi_gr(ir-2*id) ; f2  =   chi_fr(ir-2*id)
      g3  =   chi_gr(ir-3*id) ; f3  =   chi_fr(ir-3*id)
      dg0 = dxchi_gr(ir     ) ; df0 = dxchi_fr(ir     )
      dg1 = dxchi_gr(ir-1*id) ; df1 = dxchi_fr(ir-1*id)
      dg2 = dxchi_gr(ir-2*id) ; df2 = dxchi_fr(ir-2*id)
      dg3 = dxchi_gr(ir-3*id) ; df3 = dxchi_fr(ir-3*id)
   end select
   g9 = g0 + dx24 * (55.d0*dg0 - 59.d0*dg1 + 37.d0*dg2 - 9.d0*dg3)
   f9 = f0 + dx24 * (55.d0*df0 - 59.d0*df1 + 37.d0*df2 - 9.d0*df3)
   do iec = 1,nec  
      select case (is_calc)
      case (NONREL)
         call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff(ir+id,ispin,ll_core))
      case (SREL)
         call dgdx_dfdx_srel(rpos(ir+id),g9,f9,dg9,df9, &
                             ll,ee,veff(ir+id,ispin,ll_core))
      case (REL)
         call dgdx_dfdx_rel(rpos(ir+id),g9,f9,dg9,df9, &
                            kk,ee,veff(ir+id,ispin,ll_core))
      end select
      g9 = g0 + dx720 * (251.d0*dg9 + 646.d0*dg0 - 264.d0*dg1 &
                      + 106.d0*dg2 - 19.d0*dg3)
      f9 = f0 + dx720 * (251.d0*df9 + 646.d0*df0 - 264.d0*df1 &
                      + 106.d0*df2 - 19.d0*df3)
   end do   
   select case (is_calc)
   case (NONREL)
      call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                            ll,ee,veff(ir+id,ispin,ll_core))
   case (SREL)
      call dgdx_dfdx_srel(rpos(ir+id),g9,f9,dg9,df9, &
                          ll,ee,veff(ir+id,ispin,ll_core))
   case (REL)
      call dgdx_dfdx_rel(rpos(ir+id),g9,f9,dg9,df9, &
                         kk,ee,veff(ir+id,ispin,ll_core))
   end select
   select case (id)
   case(+1)
        chi_gl(ir+id) =  g9 ;   chi_fl(ir+id) =  f9
      dxchi_gl(ir+id) = dg9 ; dxchi_fl(ir+id) = df9
   case(-1)
        chi_gr(ir+id) =  g9 ;   chi_fr(ir+id) =  f9
      dxchi_gr(ir+id) = dg9 ; dxchi_fr(ir+id) = df9
   end select
   end subroutine pc_adams_6

!=====================================================================
   subroutine pc_adams_7(ier,ir,id,ll,kk,ee,nec,ispin)
!===================================================================== 
!
!  Predicts & Corrects chi_g(ir+id) & chi_f(ir+id)
!  using values at xpos(ir), xpos(ir-id), xpos(ir-2*id), 
!                  xpos(ir-3*id), xpos(ir-4*id)
!  by the Adams Predictor-Corrector Method
!     
!  M. Okamoto
!
!---------------------------------------------------------------------
   use parameters   
   implicit none
   integer,intent(in)    :: ir, ll, kk, nec, ispin
   real(8),intent(in)    :: ee
   integer,intent(inout) :: id
   integer,intent(out)   :: ier
   integer :: iec, ll_core
   real(8) :: dx720, dx1440, &
              g0, g1, g2, g3, g4, dg0, dg1, dg2, dg3, dg4, g9, dg9, &
              f0, f1, f2, f3, f4, df0, df1, df2, df3, df4, f9, df9
   ier = 0
   if (is_core == PATOM) then
      ll_core = ll
   else
      ll_core = lmax_core
   end if
   id     = sign(1, id)
   dx720  = dx/ 720.d0*dble(id)
   dx1440 = dx/1440.d0*dble(id)
   select case (id)
   case(+1)
      g0  =   chi_gl(ir     ) ; f0  =   chi_fl(ir     )
      g1  =   chi_gl(ir-1*id) ; f1  =   chi_fl(ir-1*id)
      g2  =   chi_gl(ir-2*id) ; f2  =   chi_fl(ir-2*id)
      g3  =   chi_gl(ir-3*id) ; f3  =   chi_fl(ir-3*id)
      g4  =   chi_gl(ir-4*id) ; f4  =   chi_fl(ir-4*id)
      dg0 = dxchi_gl(ir     ) ; df0 = dxchi_fl(ir     )
      dg1 = dxchi_gl(ir-1*id) ; df1 = dxchi_fl(ir-1*id)
      dg2 = dxchi_gl(ir-2*id) ; df2 = dxchi_fl(ir-2*id)
      dg3 = dxchi_gl(ir-3*id) ; df3 = dxchi_fl(ir-3*id)
      dg4 = dxchi_gl(ir-4*id) ; df4 = dxchi_fl(ir-4*id)
   case(-1)
      g0  =   chi_gr(ir     ) ; f0  =   chi_fr(ir     )
      g1  =   chi_gr(ir-1*id) ; f1  =   chi_fr(ir-1*id)
      g2  =   chi_gr(ir-2*id) ; f2  =   chi_fr(ir-2*id)
      g3  =   chi_gr(ir-3*id) ; f3  =   chi_fr(ir-3*id)
      g4  =   chi_gr(ir-4*id) ; f4  =   chi_fr(ir-4*id)
      dg0 = dxchi_gr(ir     ) ; df0 = dxchi_fr(ir     )
      dg1 = dxchi_gr(ir-1*id) ; df1 = dxchi_fr(ir-1*id)
      dg2 = dxchi_gr(ir-2*id) ; df2 = dxchi_fr(ir-2*id)
      dg3 = dxchi_gr(ir-3*id) ; df3 = dxchi_fr(ir-3*id)
      dg4 = dxchi_gr(ir-4*id) ; df4 = dxchi_fr(ir-4*id)
   end select
   g9 = g0 + dx720 * (1901.d0*dg0-2774.d0*dg1+2616.d0*dg2 &
                     -1274.d0*dg3 +251.d0*dg4)
   f9 = f0 + dx720 * (1901.d0*df0-2774.d0*df1+2616.d0*df2 &
                     -1274.d0*df3 +251.d0*df4)
   do iec = 1,nec  
      select case (is_calc)
      case (NONREL)
         call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff(ir+id,ispin,ll_core))
      case (SREL)
         call dgdx_dfdx_srel(rpos(ir+id),g9,f9,dg9,df9, &
                             ll,ee,veff(ir+id,ispin,ll_core))
      case (REL)
         call dgdx_dfdx_rel(rpos(ir+id),g9,f9,dg9,df9, &
                            kk,ee,veff(ir+id,ispin,ll_core))
      end select
      g9 = g0 + dx1440 * (475.d0*dg9+1427.d0*dg0-798.d0*dg1 &
                         +482.d0*dg2 -173.d0*dg3 +27.d0*dg4)
      f9 = f0 + dx1440 * (475.d0*df9+1427.d0*df0-798.d0*df1 &
                         +482.d0*df2 -173.d0*df3 +27.d0*df4)
   end do  
   select case (is_calc)
   case (NONREL)
      call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                            ll,ee,veff(ir+id,ispin,ll_core))
   case (SREL)
      call dgdx_dfdx_srel(rpos(ir+id),g9,f9,dg9,df9, &
                          ll,ee,veff(ir+id,ispin,ll_core))
   case (REL)
      call dgdx_dfdx_rel(rpos(ir+id),g9,f9,dg9,df9, &
                         kk,ee,veff(ir+id,ispin,ll_core))
   end select
   select case (id)
   case(+1)
      chi_gl  (ir+id) = g9  ; chi_fl  (ir+id) = f9
      dxchi_gl(ir+id) = dg9 ; dxchi_fl(ir+id) = df9
   case(-1)
      chi_gr  (ir+id) = g9  ; chi_fr  (ir+id) = f9
      dxchi_gr(ir+id) = dg9 ; dxchi_fr(ir+id) = df9
   end select
   end subroutine pc_adams_7

!=====================================================================
   subroutine pc_adams_8(ier,ir,id,ll,kk,ee,nec,ispin)
!===================================================================== 
!
!  Predicts & Corrects chi_g(ir+id) & chi_f(ir+id)
!  using values at xpos(ir), xpos(ir-id), xpos(ir-2*id), 
!                  xpos(ir-3*id), xpos(ir-4*id), xpos(ir-5*id)
!  by Adams Predictor-Corrector Method
!     
!  M. Okamoto
!
!---------------------------------------------------------------------
   use parameters   
   implicit none
   integer,intent(in)    :: ir, ll, kk, nec, ispin
   real(8),intent(in)    :: ee
   integer,intent(inout) :: id
   integer,intent(out)   :: ier
   integer :: iec, ll_core
   real(8) :: dx1440, dx60480, &
              g0, g1, g2, g3, g4, g5, dg0, dg1, dg2, dg3, dg4, dg5, &
              g9, dg9, &
              f0, f1, f2, f3, f4, f5, df0, df1, df2, df3, df4, df5, &
              f9, df9
   ier = 0
   if (is_core == PATOM) then
      ll_core = ll
   else
      ll_core = lmax_core
   end if
   id      = sign(1, id)
   dx1440  = dx/ 1440.d0 * dble(id)
   dx60480 = dx/60480.d0 * dble(id)
   select case (id)
   case(+1)
      g0  =   chi_gl(ir     ) ; f0  =   chi_fl(ir     )
      g1  =   chi_gl(ir-1*id) ; f1  =   chi_fl(ir-1*id)
      g2  =   chi_gl(ir-2*id) ; f2  =   chi_fl(ir-2*id)
      g3  =   chi_gl(ir-3*id) ; f3  =   chi_fl(ir-3*id)
      g4  =   chi_gl(ir-4*id) ; f4  =   chi_fl(ir-4*id)
      g5  =   chi_gl(ir-5*id) ; f5  =   chi_fl(ir-5*id)
      dg0 = dxchi_gl(ir     ) ; df0 = dxchi_fl(ir     )
      dg1 = dxchi_gl(ir-1*id) ; df1 = dxchi_fl(ir-1*id)
      dg2 = dxchi_gl(ir-2*id) ; df2 = dxchi_fl(ir-2*id)
      dg3 = dxchi_gl(ir-3*id) ; df3 = dxchi_fl(ir-3*id)
      dg4 = dxchi_gl(ir-4*id) ; df4 = dxchi_fl(ir-4*id)
      dg5 = dxchi_gl(ir-5*id) ; df5 = dxchi_fl(ir-5*id)
   case(-1)
      g0  =   chi_gr(ir     ) ; f0  =   chi_fr(ir     )
      g1  =   chi_gr(ir-1*id) ; f1  =   chi_fr(ir-1*id)
      g2  =   chi_gr(ir-2*id) ; f2  =   chi_fr(ir-2*id)
      g3  =   chi_gr(ir-3*id) ; f3  =   chi_fr(ir-3*id)
      g4  =   chi_gr(ir-4*id) ; f4  =   chi_fr(ir-4*id)
      g5  =   chi_gr(ir-5*id) ; f5  =   chi_fr(ir-5*id)
      dg0 = dxchi_gr(ir     ) ; df0 = dxchi_fr(ir     )
      dg1 = dxchi_gr(ir-1*id) ; df1 = dxchi_fr(ir-1*id)
      dg2 = dxchi_gr(ir-2*id) ; df2 = dxchi_fr(ir-2*id)
      dg3 = dxchi_gr(ir-3*id) ; df3 = dxchi_fr(ir-3*id)
      dg4 = dxchi_gr(ir-4*id) ; df4 = dxchi_fr(ir-4*id)
      dg5 = dxchi_gr(ir-5*id) ; df5 = dxchi_fr(ir-5*id)
   end select
   g9 = g0 + dx1440 * (4277.d0*dg0 - 7923.d0*dg1 + 9982.d0*dg2 &
                      -7298.d0*dg3 + 2877.d0*dg4 -  475.d0*dg5)
   f9 = f0 + dx1440 * (4277.d0*df0 - 7923.d0*df1 + 9982.d0*df2 &
                      -7298.d0*df3 + 2877.d0*df4 -  475.d0*df5)
   do iec = 1,nec 
      select case (is_calc)
      case (NONREL)
         call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff(ir+id,ispin,ll_core))
      case (SREL)
         call dgdx_dfdx_srel(rpos(ir+id),g9,f9,dg9,df9, &
                             ll,ee,veff(ir+id,ispin,ll_core))
      case (REL)
         call dgdx_dfdx_rel(rpos(ir+id),g9,f9,dg9,df9, &
                            kk,ee,veff(ir+id,ispin,ll_core))
      end select
      g9 = g0 + dx60480 * (19087.d0*dg9+65112.d0*dg0-46461.d0*dg1 &
                          +37504.d0*dg2+20211.d0*dg3 +6312.d0*dg4 &
                          -863.d0*dg5)
      f9 = f0 + dx60480 * (19087.d0*df9+65112.d0*df0-46461.d0*df1 &
                          +37504.d0*df2+20211.d0*df3 +6312.d0*df4 &
                          -863.d0*df5)
   end do   
   select case (is_calc)
   case (NONREL)
      call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                            ll,ee,veff(ir+id,ispin,ll_core))
   case (SREL)
      call dgdx_dfdx_srel(rpos(ir+id),g9,f9,dg9,df9, &
                          ll,ee,veff(ir+id,ispin,ll_core))
   case (REL)
      call dgdx_dfdx_rel(rpos(ir+id),g9,f9,dg9,df9, &
                         kk,ee,veff(ir+id,ispin,ll_core))
   end select
   select case (id)
   case(+1)
      chi_gl  (ir+id) = g9  ; chi_fl  (ir+id) = f9
      dxchi_gl(ir+id) = dg9 ; dxchi_fl(ir+id) = df9
   case(-1)
      chi_gr  (ir+id) = g9  ; chi_fr  (ir+id) = f9
      dxchi_gr(ir+id) = dg9 ; dxchi_fr(ir+id) = df9
   end select
   end subroutine pc_adams_8

!=====================================================================
   subroutine pc_adams_4ps(ier,ir,id,ll,tt,kk,ee,nec,ips)
!===================================================================== 
!
!  Predicts & Corrects chi_g(ir+id) & chi_f(ir+id)
!  using values at xpos(ir), xpos(ir-id), xpos(ir-2*id)
!  by the Adams Predictor-Corrector Method (3rd order)
!     
!  M. Okamoto
!
!---------------------------------------------------------------------
   use parameters   
   implicit none
   integer,intent(in)    :: ir, ll, tt, kk, nec, ips
   real(8),intent(in)    :: ee
   integer,intent(inout) :: id
   integer,intent(out)   :: ier
   integer :: iec
   real(8) :: dx12, &
              g0, g1, g2, dg0, dg1, dg2, g9, dg9, &
              f0, f1, f2, df0, df1, df2, f9, df9
   ier = 0
   id   = sign(1, id)
   dx12 = dx / 12.d0 * dble(id)
   select case (id)
   case(+1)
      g0  =   chi_gl(ir     ) ; f0  =   chi_fl(ir     )
      g1  =   chi_gl(ir-1*id) ; f1  =   chi_fl(ir-1*id)
      g2  =   chi_gl(ir-2*id) ; f2  =   chi_fl(ir-2*id)
      dg0 = dxchi_gl(ir     ) ; df0 = dxchi_fl(ir     )
      dg1 = dxchi_gl(ir-1*id) ; df1 = dxchi_fl(ir-1*id)
      dg2 = dxchi_gl(ir-2*id) ; df2 = dxchi_fl(ir-2*id)
   case(-1)
      g0  =   chi_gr(ir     ) ; f0  =   chi_fr(ir     )
      g1  =   chi_gr(ir-1*id) ; f1  =   chi_fr(ir-1*id)
      g2  =   chi_gr(ir-2*id) ; f2  =   chi_fr(ir-2*id)
      dg0 = dxchi_gr(ir     ) ; df0 = dxchi_fr(ir     )
      dg1 = dxchi_gr(ir-1*id) ; df1 = dxchi_fr(ir-1*id)
      dg2 = dxchi_gr(ir-2*id) ; df2 = dxchi_fr(ir-2*id)
   end select
   g9 = g0 + dx12 * (23.d0*dg0 - 16.d0*dg1 + 5.d0*dg2)
   f9 = f0 + dx12 * (23.d0*df0 - 16.d0*df1 + 5.d0*df2)
   do iec = 1,nec  
      call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff_ps(ir+id,ips,1))
      g9 = g0 + dx12 * (5.d0*dg9 + 8.d0*dg0 - dg1)
      f9 = f0 + dx12 * (5.d0*df9 + 8.d0*df0 - df1)
   end do    
      call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff_ps(ir+id,ips,1))
   select case (id)
   case(+1)
        chi_gl(ir+id) =  g9 ;   chi_fl(ir+id) =  f9
      dxchi_gl(ir+id) = dg9 ; dxchi_fl(ir+id) = df9
   case(-1)
        chi_gr(ir+id) =  g9 ;   chi_fr(ir+id) =  f9
      dxchi_gr(ir+id) = dg9 ; dxchi_fr(ir+id) = df9
   end select
   end subroutine pc_adams_4ps

!=====================================================================
   subroutine pc_adams_5ps(ier,ir,id,ll,tt,kk,ee,nec,ips)
!=====================================================================
!
!  Predicts & Corrects chi_g(ir+id) & chi_f(ir+id)
!  using values at xpos(ir), xpos(ir-id), xpos(ir-2*id), xpos(ir-3*id)
!  by the Adams Predictor-Corrector Method
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   use parameters   
   implicit none
   integer,intent(in)    :: ir, ll, tt, kk, nec, ips
   real(8),intent(in)    :: ee
   integer,intent(inout) :: id
   integer,intent(out)   :: ier
   integer :: iec
   real(8) :: dx24, &
              g0, g1, g2, g3, dg0, dg1, dg2, dg3, g9, dg9, &
              f0, f1, f2, f3, df0, df1, df2, df3, f9, df9
   ier = 0
   id   = sign(1, id)
   dx24 = dx / 24.d0 * dble(id)
   select case (id)
   case(+1)
      g0  =   chi_gl(ir     ) ; f0  =   chi_fl(ir     )
      g1  =   chi_gl(ir-1*id) ; f1  =   chi_fl(ir-1*id)
      g2  =   chi_gl(ir-2*id) ; f2  =   chi_fl(ir-2*id)
      g3  =   chi_gl(ir-3*id) ; f3  =   chi_fl(ir-3*id)
      dg0 = dxchi_gl(ir     ) ; df0 = dxchi_fl(ir     )
      dg1 = dxchi_gl(ir-1*id) ; df1 = dxchi_fl(ir-1*id)
      dg2 = dxchi_gl(ir-2*id) ; df2 = dxchi_fl(ir-2*id)
      dg3 = dxchi_gl(ir-3*id) ; df3 = dxchi_fl(ir-3*id)
   case(-1)
      g0  =   chi_gr(ir     ) ; f0  =   chi_fr(ir     )
      g1  =   chi_gr(ir-1*id) ; f1  =   chi_fr(ir-1*id)
      g2  =   chi_gr(ir-2*id) ; f2  =   chi_fr(ir-2*id)
      g3  =   chi_gr(ir-3*id) ; f3  =   chi_fr(ir-3*id)
      dg0 = dxchi_gr(ir     ) ; df0 = dxchi_fr(ir     )
      dg1 = dxchi_gr(ir-1*id) ; df1 = dxchi_fr(ir-1*id)
      dg2 = dxchi_gr(ir-2*id) ; df2 = dxchi_fr(ir-2*id)
      dg3 = dxchi_gr(ir-3*id) ; df3 = dxchi_fr(ir-3*id)
   end select
   g9 = g0 + dx24 * (55.d0*dg0 - 59.d0*dg1 + 37.d0*dg2 - 9.d0*dg3)
   f9 = f0 + dx24 * (55.d0*df0 - 59.d0*df1 + 37.d0*df2 - 9.d0*df3)
   do iec = 1,nec  
      call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff_ps(ir+id,ips,1))
      g9 = g0 + dx24 * (9.d0*dg9 + 19.d0*dg0 - 5.d0*dg1 + dg2)
      f9 = f0 + dx24 * (9.d0*df9 + 19.d0*df0 - 5.d0*df1 + df2)
   end do  
   call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                            ll,ee,veff_ps(ir+id,ips,1))
   select case (id)
   case(+1)
        chi_gl(ir+id) =  g9 ;   chi_fl(ir+id) =  f9
      dxchi_gl(ir+id) = dg9 ; dxchi_fl(ir+id) = df9
   case(-1)
        chi_gr(ir+id) =  g9 ;   chi_fr(ir+id) =  f9
      dxchi_gr(ir+id) = dg9 ; dxchi_fr(ir+id) = df9
   end select
   end subroutine pc_adams_5ps

!=====================================================================
   subroutine pc_adams_6ps(ier,ir,id,ll,tt,kk,ee,nec,ips)
!=====================================================================
!
!  Predicts & Corrects chi_g(ir+id) & chi_f(ir+id)
!  using values at xpos(ir), xpos(ir-id), xpos(ir-2*id), xpos(ir-3*id)
!  by the Adams Predictor-Corrector Method
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   use parameters   
   implicit none
   integer,intent(in)    :: ir, ll, tt, kk, nec, ips
   real(8),intent(in)    :: ee
   integer,intent(inout) :: id
   integer,intent(out)   :: ier
   integer :: iec
   real(8) :: dx24, dx720, &
              g0, g1, g2, g3, dg0, dg1, dg2, dg3, g9, dg9, &
              f0, f1, f2, f3, df0, df1, df2, df3, f9, df9
   ier = 0
   id    = sign(1, id)
   dx24  = dx /  24.d0 * dble(id)
   dx720 = dx / 720.d0 * dble(id)
   select case (id)
   case(+1)
      g0  =   chi_gl(ir     ) ; f0  =   chi_fl(ir     )
      g1  =   chi_gl(ir-1*id) ; f1  =   chi_fl(ir-1*id)
      g2  =   chi_gl(ir-2*id) ; f2  =   chi_fl(ir-2*id)
      g3  =   chi_gl(ir-3*id) ; f3  =   chi_fl(ir-3*id)
      dg0 = dxchi_gl(ir     ) ; df0 = dxchi_fl(ir     )
      dg1 = dxchi_gl(ir-1*id) ; df1 = dxchi_fl(ir-1*id)
      dg2 = dxchi_gl(ir-2*id) ; df2 = dxchi_fl(ir-2*id)
      dg3 = dxchi_gl(ir-3*id) ; df3 = dxchi_fl(ir-3*id)
   case(-1)
      g0  =   chi_gr(ir     ) ; f0  =   chi_fr(ir     )
      g1  =   chi_gr(ir-1*id) ; f1  =   chi_fr(ir-1*id)
      g2  =   chi_gr(ir-2*id) ; f2  =   chi_fr(ir-2*id)
      g3  =   chi_gr(ir-3*id) ; f3  =   chi_fr(ir-3*id)
      dg0 = dxchi_gr(ir     ) ; df0 = dxchi_fr(ir     )
      dg1 = dxchi_gr(ir-1*id) ; df1 = dxchi_fr(ir-1*id)
      dg2 = dxchi_gr(ir-2*id) ; df2 = dxchi_fr(ir-2*id)
      dg3 = dxchi_gr(ir-3*id) ; df3 = dxchi_fr(ir-3*id)
   end select
   g9 = g0 + dx24 * (55.d0*dg0 - 59.d0*dg1 + 37.d0*dg2 - 9.d0*dg3)
   f9 = f0 + dx24 * (55.d0*df0 - 59.d0*df1 + 37.d0*df2 - 9.d0*df3)
   do iec = 1,nec 
      call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff_ps(ir+id,ips,1))
      g9 = g0 + dx720 * (251.d0*dg9 + 646.d0*dg0 - 264.d0*dg1 &
                      + 106.d0*dg2 - 19.d0*dg3)
      f9 = f0 + dx720 * (251.d0*df9 + 646.d0*df0 - 264.d0*df1 &
                      + 106.d0*df2 - 19.d0*df3)
   end do   
   call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                            ll,ee,veff_ps(ir+id,ips,1))
   select case (id)
   case(+1)
        chi_gl(ir+id) =  g9 ;   chi_fl(ir+id) =  f9
      dxchi_gl(ir+id) = dg9 ; dxchi_fl(ir+id) = df9
   case(-1)
        chi_gr(ir+id) =  g9 ;   chi_fr(ir+id) =  f9
      dxchi_gr(ir+id) = dg9 ; dxchi_fr(ir+id) = df9
   end select
   end subroutine pc_adams_6ps

!=====================================================================
   subroutine pc_adams_7ps(ier,ir,id,ll,tt,kk,ee,nec,ips)
!===================================================================== 
!
!  Predicts & Corrects chi_g(ir+id) & chi_f(ir+id)
!  using values at xpos(ir), xpos(ir-id), xpos(ir-2*id), 
!                  xpos(ir-3*id), xpos(ir-4*id)
!  by the Adams Predictor-Corrector Method
!     
!  M. Okamoto
!
!---------------------------------------------------------------------
   use parameters   
   implicit none
   integer,intent(in)    :: ir, ll, tt, kk, nec, ips
   real(8),intent(in)    :: ee
   integer,intent(inout) :: id
   integer,intent(out)   :: ier
   integer :: iec
   real(8) :: dx720, dx1440, &
              g0, g1, g2, g3, g4, dg0, dg1, dg2, dg3, dg4, g9, dg9, &
              f0, f1, f2, f3, f4, df0, df1, df2, df3, df4, f9, df9
   ier = 0
   id     = sign(1, id)
   dx720  = dx/ 720.d0*dble(id)
   dx1440 = dx/1440.d0*dble(id)
   select case (id)
   case(+1)
      g0  =   chi_gl(ir     ) ; f0  =   chi_fl(ir     )
      g1  =   chi_gl(ir-1*id) ; f1  =   chi_fl(ir-1*id)
      g2  =   chi_gl(ir-2*id) ; f2  =   chi_fl(ir-2*id)
      g3  =   chi_gl(ir-3*id) ; f3  =   chi_fl(ir-3*id)
      g4  =   chi_gl(ir-4*id) ; f4  =   chi_fl(ir-4*id)
      dg0 = dxchi_gl(ir     ) ; df0 = dxchi_fl(ir     )
      dg1 = dxchi_gl(ir-1*id) ; df1 = dxchi_fl(ir-1*id)
      dg2 = dxchi_gl(ir-2*id) ; df2 = dxchi_fl(ir-2*id)
      dg3 = dxchi_gl(ir-3*id) ; df3 = dxchi_fl(ir-3*id)
      dg4 = dxchi_gl(ir-4*id) ; df4 = dxchi_fl(ir-4*id)
   case(-1)
      g0  =   chi_gr(ir     ) ; f0  =   chi_fr(ir     )
      g1  =   chi_gr(ir-1*id) ; f1  =   chi_fr(ir-1*id)
      g2  =   chi_gr(ir-2*id) ; f2  =   chi_fr(ir-2*id)
      g3  =   chi_gr(ir-3*id) ; f3  =   chi_fr(ir-3*id)
      g4  =   chi_gr(ir-4*id) ; f4  =   chi_fr(ir-4*id)
      dg0 = dxchi_gr(ir     ) ; df0 = dxchi_fr(ir     )
      dg1 = dxchi_gr(ir-1*id) ; df1 = dxchi_fr(ir-1*id)
      dg2 = dxchi_gr(ir-2*id) ; df2 = dxchi_fr(ir-2*id)
      dg3 = dxchi_gr(ir-3*id) ; df3 = dxchi_fr(ir-3*id)
      dg4 = dxchi_gr(ir-4*id) ; df4 = dxchi_fr(ir-4*id)
   end select
   g9 = g0 + dx720 * (1901.d0*dg0-2774.d0*dg1+2616.d0*dg2 &
                     -1274.d0*dg3 +251.d0*dg4)
   f9 = f0 + dx720 * (1901.d0*df0-2774.d0*df1+2616.d0*df2 &
                     -1274.d0*df3 +251.d0*df4)
   do iec = 1,nec 
      call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff_ps(ir+id,ips,1))
      g9 = g0 + dx1440 * (475.d0*dg9+1427.d0*dg0-798.d0*dg1 &
                         +482.d0*dg2 -173.d0*dg3 +27.d0*dg4)
      f9 = f0 + dx1440 * (475.d0*df9+1427.d0*df0-798.d0*df1 &
                         +482.d0*df2 -173.d0*df3 +27.d0*df4)
   end do    
   call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                            ll,ee,veff_ps(ir+id,ips,1))
   select case (id)
   case(+1)
      chi_gl  (ir+id) = g9  ; chi_fl  (ir+id) = f9
      dxchi_gl(ir+id) = dg9 ; dxchi_fl(ir+id) = df9
   case(-1)
      chi_gr  (ir+id) = g9  ; chi_fr  (ir+id) = f9
      dxchi_gr(ir+id) = dg9 ; dxchi_fr(ir+id) = df9
   end select
   end subroutine pc_adams_7ps

!=====================================================================
   subroutine pc_adams_8ps(ier,ir,id,ll,tt,kk,ee,nec,ips)
!===================================================================== 
!
!  Predicts & Corrects chi_g(ir+id) & chi_f(ir+id)
!  using values at xpos(ir), xpos(ir-id), xpos(ir-2*id), 
!                  xpos(ir-3*id), xpos(ir-4*id), xpos(ir-5*id)
!  by Adams Predictor-Corrector Method
!     
!  M. Okamoto
!
!---------------------------------------------------------------------
   use parameters   
   implicit none
   integer,intent(in)    :: ir, ll, tt, kk, nec, ips
   real(8),intent(in)    :: ee
   integer,intent(inout) :: id
   integer,intent(out)   :: ier
   integer :: iec
   real(8) :: dx1440, dx60480, &
              g0, g1, g2, g3, g4, g5, dg0, dg1, dg2, dg3, dg4, dg5, &
              g9, dg9, &
              f0, f1, f2, f3, f4, f5, df0, df1, df2, df3, df4, df5, &
              f9, df9
   ier = 0
   id      = sign(1, id)
   dx1440  = dx/ 1440.d0 * dble(id)
   dx60480 = dx/60480.d0 * dble(id)
   select case (id)
   case(+1)
      g0  =   chi_gl(ir     ) ; f0  =   chi_fl(ir     )
      g1  =   chi_gl(ir-1*id) ; f1  =   chi_fl(ir-1*id)
      g2  =   chi_gl(ir-2*id) ; f2  =   chi_fl(ir-2*id)
      g3  =   chi_gl(ir-3*id) ; f3  =   chi_fl(ir-3*id)
      g4  =   chi_gl(ir-4*id) ; f4  =   chi_fl(ir-4*id)
      g5  =   chi_gl(ir-5*id) ; f5  =   chi_fl(ir-5*id)
      dg0 = dxchi_gl(ir     ) ; df0 = dxchi_fl(ir     )
      dg1 = dxchi_gl(ir-1*id) ; df1 = dxchi_fl(ir-1*id)
      dg2 = dxchi_gl(ir-2*id) ; df2 = dxchi_fl(ir-2*id)
      dg3 = dxchi_gl(ir-3*id) ; df3 = dxchi_fl(ir-3*id)
      dg4 = dxchi_gl(ir-4*id) ; df4 = dxchi_fl(ir-4*id)
      dg5 = dxchi_gl(ir-5*id) ; df5 = dxchi_fl(ir-5*id)
   case(-1)
      g0  =   chi_gr(ir     ) ; f0  =   chi_fr(ir     )
      g1  =   chi_gr(ir-1*id) ; f1  =   chi_fr(ir-1*id)
      g2  =   chi_gr(ir-2*id) ; f2  =   chi_fr(ir-2*id)
      g3  =   chi_gr(ir-3*id) ; f3  =   chi_fr(ir-3*id)
      g4  =   chi_gr(ir-4*id) ; f4  =   chi_fr(ir-4*id)
      g5  =   chi_gr(ir-5*id) ; f5  =   chi_fr(ir-5*id)
      dg0 = dxchi_gr(ir     ) ; df0 = dxchi_fr(ir     )
      dg1 = dxchi_gr(ir-1*id) ; df1 = dxchi_fr(ir-1*id)
      dg2 = dxchi_gr(ir-2*id) ; df2 = dxchi_fr(ir-2*id)
      dg3 = dxchi_gr(ir-3*id) ; df3 = dxchi_fr(ir-3*id)
      dg4 = dxchi_gr(ir-4*id) ; df4 = dxchi_fr(ir-4*id)
      dg5 = dxchi_gr(ir-5*id) ; df5 = dxchi_fr(ir-5*id)
   end select
   g9 = g0 + dx1440 * (4277.d0*dg0 - 7923.d0*dg1 + 9982.d0*dg2 &
                      -7298.d0*dg3 + 2877.d0*dg4 -  475.d0*dg5)
   f9 = f0 + dx1440 * (4277.d0*df0 - 7923.d0*df1 + 9982.d0*df2 &
                      -7298.d0*df3 + 2877.d0*df4 -  475.d0*df5)
   do iec = 1,nec 
      call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                               ll,ee,veff_ps(ir+id,ips,1))
      g9 = g0 + dx60480 * (19087.d0*dg9+65112.d0*dg0-46461.d0*dg1 &
                          +37504.d0*dg2+20211.d0*dg3 +6312.d0*dg4 &
                          -863.d0*dg5)
      f9 = f0 + dx60480 * (19087.d0*df9+65112.d0*df0-46461.d0*df1 &
                          +37504.d0*df2+20211.d0*df3 +6312.d0*df4 &
                          -863.d0*df5)
   end do   
   call dgdx_dfdx_nonrel(rpos(ir+id),g9,f9,dg9,df9, &
                            ll,ee,veff_ps(ir+id,ips,1))
   select case (id)
   case(+1)
      chi_gl  (ir+id) = g9  ; chi_fl  (ir+id) = f9
      dxchi_gl(ir+id) = dg9 ; dxchi_fl(ir+id) = df9
   case(-1)
      chi_gr  (ir+id) = g9  ; chi_fr  (ir+id) = f9
      dxchi_gr(ir+id) = dg9 ; dxchi_fr(ir+id) = df9
   end select
   end subroutine pc_adams_8ps
