! ************************************************************* 
!
!   This is a software package CIAO
!
!     developed as a part of the national project "Research and 
!     Development of Innovative Simulation software",which is   
!     supported by the next-generation IT program of MEXT of Japan
!
!   latest version: 
!
!     4.0:  2013/01/17 
!           codes for spin-polarized pseudopotential generation are added
!
! ************************************************************* 
!
!  Software name : CIAO (Code for Investigating Atomic Orbitals)
!  Subroutine(s) : gga_fxc_rmesh, gga_pbe96_fxc, gga_pbe96_spherical
!                : gga_pw91_fxc
!  Function(s)   : fn_ex0, fn_fx_pbe96,
!                : fn_fx1_pbe96, fn_fxp_pbe96, fn_fx2_pbe96,
!                : fn_fx_pw91, fn_fxp_pw91, fn_c_pw91, fn_cp_pw91
!  Author(s)     : Masakuni Okamoto (August 25, 2003)
!
!  The license of the code and contact address :
!  See the files, COPYRIGHT and LICENSE (or LICENSE_J.pdf)
!
!=====================================================================
   subroutine gga_fxc_rmesh( &
      ier,ifile,is_xc_type,iord_diff,nmesh,nspin,        &
      rpos,wr,rho1,rho2,drho1,drho2,                     &
      fx,fc,dfx,dfc,dfxda,dfcda,vx,vc,ex,ec,ex_sum,ec_sum)
!=====================================================================
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   integer,intent(in)  :: ifile, is_xc_type, iord_diff, &
                          nmesh, nspin
   real(8),intent(in)  :: rpos(nmesh), wr(nmesh),    &
                          rho1(nmesh), rho2(nmesh),  &
                          drho1(nmesh), drho2(nmesh)
   integer,intent(out) :: ier
   real(8),intent(out) :: fx(nmesh), fc(nmesh),             &
                          dfx(nmesh,2), dfc(nmesh,2),       &
                          dfxda(nmesh,2), dfcda(nmesh),     &
                          vx(nmesh,nspin), vc(nmesh,nspin), &
                          ex(nmesh), ec(nmesh),             &
                          ex_sum, ec_sum
   integer :: ir
   real(8) :: fx_tmp, fc_tmp, dfx_tmp(2), dfc_tmp(2), &
              dfxda_tmp(2), dfcda_tmp, rho_tmp
   real(8),parameter :: &
      PI = 3.1415926535897932384626433832795028d0, &
      EPS = 1.d-25
   ier = 0
   fx(:) = 0.d0 ; fc(:) = 0.d0 ; dfx(:,:) = 0.d0 ; dfc(:,:) = 0.d0
   dfxda(:,:) = 0.d0 ; dfcda(:) = 0.d0
   vx(:,:) = 0.d0 ; vc(:,:) = 0.d0 ; ex(:) = 0.d0 ; ec(:) = 0.d0
   ex_sum = 0.d0 ; ec_sum = 0.d0
   select case (is_xc_type)
   case (1,2)
      do ir = 1,nmesh
         call gga_pbe96_fxc(is_xc_type-1, &
            rho1(ir),rho2(ir),drho1(ir),drho2(ir),drho1(ir)+drho2(ir),&
            fx_tmp,fc_tmp,dfx_tmp,dfc_tmp,dfxda_tmp,dfcda_tmp)
         fx(ir) = fx_tmp            ; fc(ir) = fc_tmp
         dfx(ir,:) = dfx_tmp(:)     ; dfc(ir,:) = dfc_tmp(:)
         dfxda(ir,:) = dfxda_tmp(:) ; dfcda(ir) = dfcda_tmp
      end do
   case (3)
      do ir = 1,nmesh
         call gga_pw91_fxc( &
            rho1(ir),rho2(ir),drho1(ir),drho2(ir),drho1(ir)+drho2(ir),&
            fx_tmp,fc_tmp,dfx_tmp,dfc_tmp,dfxda_tmp,dfcda_tmp)
         fx(ir) = fx_tmp            ; fc(ir) = fc_tmp
         dfx(ir,:) = dfx_tmp(:)     ; dfc(ir,:) = dfc_tmp(:)
         dfxda(ir,:) = dfxda_tmp(:) ; dfcda(ir) = dfcda_tmp
      end do
   case default
      write(ifile,*) '### ERROR ### is_xc_type is wrong.'
      write(ifile,*) '   is_xc_type ...', is_xc_type
      ier = 1 ; go to 99
   end select
   do ir = 1,nmesh
      if (drho1(ir) < 0.d0) then
         dfxda(ir,1) = - dfxda(ir,1)
      end if
      if (drho2(ir) < 0.d0) then
         dfxda(ir,2) = - dfxda(ir,2)
      end if
      if (drho1(ir)+drho2(ir) < 0.d0) then
         dfcda(ir) = - dfcda(ir)
      end if
   end do
   call calc_diff_exp(ier,iord_diff,nmesh, &
      rpos,dfxda(1,1),vx(1,1))
   if (ier /= 0) then
      write(ifile,*) '### ERROR ### in calc_diff_exp'
      ier = ier + 100 ; go to 99
   end if
   if (nspin == 2) then
      call calc_diff_exp(ier,iord_diff,nmesh, &
         rpos,dfxda(1,nspin),vx(1,nspin))
      if (ier /= 0) then
         write(ifile,*) '### ERROR ### in calc_diff_exp'
         ier = ier + 200 ; go to 99
      end if
   end if
   call calc_diff_exp(ier,iord_diff,nmesh,rpos,dfcda(1),vc(1,1))
   if (ier /= 0) then
      write(ifile,*) '### ERROR ### in calc_diff_exp'
      ier = ier + 300 ; go to 99
   end if
   if (nspin == 2) then
      vc(:,nspin) = vc(:,1)
   end if
   vx(:,1) = dfx(:,1) - vx(:,1) - 2.d0/rpos(:)*dfxda(:,1)
   vc(:,1) = dfc(:,1) - vc(:,1) - 2.d0/rpos(:)*dfcda(:)
   if (nspin == 2) then
      vx(:,nspin) = dfx(:,2) - vx(:,nspin) - 2.d0/rpos(:)*dfxda(:,2)
      vc(:,nspin) = dfc(:,2) - vc(:,nspin) - 2.d0/rpos(:)*dfcda(:)
   end if
   do ir = 1,nmesh
      ex_sum = ex_sum + 4.d0*PI*rpos(ir)*rpos(ir)*wr(ir)*fx(ir)
      ec_sum = ec_sum + 4.d0*PI*rpos(ir)*rpos(ir)*wr(ir)*fc(ir)
      rho_tmp = rho1(ir) + rho2(ir)
      if (abs(rho_tmp) > EPS) then
         ex(ir) = fx(ir) / rho_tmp
         ec(ir) = fc(ir) / rho_tmp
      end if
      if (rho1(ir) < EPS) then
         vc(ir,1) = 0.d0
      end if
      if (nspin == 2) then
         if (rho2(ir) < EPS) then
            vc(ir,nspin) = 0.d0
         end if
      end if
   end do
99 continue
   end subroutine gga_fxc_rmesh

!=====================================================================
   subroutine gga_pbe96_fxc(imode, &
                 rho1_in,rho2_in,drho1_in,drho2_in,drho_in, &
                 fx_gga,fc_gga,dfx_gga,dfc_gga,dfxda_gga,dfcda_gga)
!=====================================================================
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   integer,intent(in)  :: imode
   real(8),intent(in)  :: rho1_in, rho2_in, &
                          drho1_in, drho2_in, drho_in
   real(8),intent(out) :: fx_gga, fc_gga, &
                          dfx_gga(2), dfc_gga(2), &
                          dfxda_gga(2), dfcda_gga
   integer :: ispin, spin(2)
   real(8) :: &
     rho1, rho2, drho1, drho2, &
     ex_unif, ec_unif, vx_unif(2), vc_unif(2), &
     decdr_unif, decdz_unif, &
     rho, drho, rs, zeta, dzeta, phi, dphi, &
     ks, kf, kf1, kf2, s1, s2, t, &
     a, h, q0, q1, q2, u, exp_u, &
     dq0dz, dq1dr, dq1dz, dq1dt, dq2dr, dq2dz, dq2dt, &
     dadr, dadz, dhdr, dhdz, dhdt, &
     fn_ex0, fn_fx_pbe96, fn_fx1_pbe96, fn_fxp_pbe96, &
     p0, att, t2, t3, t4, t5, t6, a2, qqq, &
     ex0, ex0_1, ex0_2, fx0_1, fx0_2, fx1_1, fx1_2, fxp_1, fxp_2, &
     z1, z2, z3, z4, ec_gga
   real(8),parameter :: &
      argmax_exp = 300.d0,                               &
      argmin_ln  = 1.d-15,                               &
      pi      = 3.1415926535897932384626433832795029d0,  &
      pipi    = 9.8696044010893586188344909998761511d0,  &
      frac1_3 = 0.33333333333333333333333333333333333d0, &
      frac2_3 = 0.66666666666666666666666666666666666d0, &
      frac4_3 = 1.33333333333333333333333333333333333d0, &
      frac7_6 = 1.16666666666666666666666666666666666d0, &
      rhomin   = 1.d-25,                                 &
      eps_zeta = 1.d-5
   real(8) :: kappa, mu_kappa
   real(8),parameter :: &
      beta       = 0.06672455060314922d0, &
      mu         = 0.21951497278705521609828723131160858d0,  &
      gamma      = 0.031090690869654895034940863712730629d0, &
      beta_gamma = 2.1461263399673646060592325913558248d0
    rho1  =  rho1_in ; rho2  =  rho2_in
    drho1 = drho1_in ; drho2 = drho2_in ; drho = drho_in
   select case (imode)
   case (0)
      kappa    = 0.8040d0
      mu_kappa = 0.27302857315259963733997193463227010d0
   case (1)
      kappa    = 1.2450d0
      mu_kappa = 0.176317247218518d0
   end select
   spin(1) = +1 ; spin(2) = -1     
   rho = rho1 + rho2
   if (rho1 < rhomin) then
      drho1 = 0.d0
   end if
   if (rho2 < rhomin) then
      drho2 = 0.d0
   end if
   if (rho < rhomin) then
      drho = 0.d0
   end if
   if ((rho1 < rhomin).and.(rho2 < rhomin)) then
      zeta  = 0.d0
      dzeta = 0.d0
      phi   = 0.5d0
      dphi  = 0.d0
   else if ((rho1 >= rhomin).and.(rho2 < rhomin)) then
      zeta  = +1.d0
      dzeta =  0.d0
      phi   = 2.d0**(-frac1_3)
      dphi  = frac1_3 * 2.d0**(-frac1_3)
   else if ((rho1 < rhomin).and.(rho2 >= rhomin)) then
      zeta  = -1.d0
      dzeta =  0.d0
      phi   = 2.d0**(-frac1_3)
      dphi  = -frac1_3 * 2.d0**(-frac1_3)
   else
      zeta  = (rho1 - rho2) / rho
      dzeta = 2.d0 * (rho2*drho1 - rho1*drho2) / (rho*rho)
      phi   = 0.5d0 &
            * ((1.d0+zeta)**( frac2_3) + (1.d0-zeta)**( frac2_3))
      dphi  = (frac1_3) &
            * ((1.d0+zeta)**(-frac1_3) - (1.d0-zeta)**(-frac1_3))
   end if
   kf1   = (6.d0*pipi*rho1)**(frac1_3)
   kf2   = (6.d0*pipi*rho2)**(frac1_3)
   kf    = (3.d0*pipi*rho )**(frac1_3)
   ks    = sqrt(4.d0*kf/pi)
   rs    = (0.75d0/pi/rho)**(frac1_3)
   if (rho1 < rhomin) then
      s1 = 0.d0
   else
      s1 = abs(drho1) / (2.d0*kf1*rho1)
   end if
   if (rho2 < rhomin) then
      s2 = 0.d0
   else
      s2 = abs(drho2) / (2.d0*kf2*rho2)
   end if
   if (rho < rhomin) then
      t = 0.d0
   else
      t = abs(drho) / (2.d0*phi*ks*rho)
   end if
   call lda_pw92(rho1,rho2,ex_unif,ec_unif,vx_unif,vc_unif, &
                 decdr_unif,decdz_unif)
   q0 = gamma * phi * phi * phi
      p0 = beta * phi * phi * phi
      t2 =  t * t ; t3 = t2 * t ; t4 = t3 * t
      t5 = t4 * t ; t6 = t5 * t
   u  = min(ec_unif/q0, -1.d-15)
      exp_u = exp(min(-u,argmax_exp))
   a  = beta_gamma / (exp_u - 1.d0)
      att = a * t2 ; a2 = a * a
   q1 = beta_gamma * t2 * (1.d0 + att)
   q2 = 1.d0 + att * (1.d0 + att)
   h  = q0 * log( max(1.d0+q1/q2, argmin_ln) )
   dadr = a2 * exp_u * decdr_unif/p0
   dadz = a2 * exp_u * 3.d0*ec_unif*dphi/(p0*phi)
   dq0dz = 3.d0 * gamma * phi * phi * dphi
   dq1dr = beta_gamma * t4 * dadr
   dq1dz = beta_gamma * t4 * dadz
   dq1dt = 2.d0 * beta_gamma * t * (1.d0 + 2.d0*att)
   dq2dr = t2 * (1.d0 + 2.d0*att) * dadr
   dq2dz = t2 * (1.d0 + 2.d0*att) * dadz
   dq2dt = 2.d0*a*t * (1.d0 + 2.d0*att)
      qqq = q0/(q1+q2)/q2
   dhdr = - qqq * beta_gamma *a*t6*(2.d0+att) * dadr 
   dhdz = dq0dz * (h/q0) - qqq * beta_gamma * a*t6*(2.d0+att) * dadz
   dhdt = qqq * 2.d0*beta_gamma * t*(1.d0+2.d0*att)
   ex0   = fn_ex0(     rho )
   ex0_1 = fn_ex0(2.d0*rho1) ; ex0_2 = fn_ex0(2.d0*rho2)
   fx0_1 = fn_fx_pbe96 (s1, kappa) ; fx0_2 = fn_fx_pbe96 (s2, kappa)
   fx1_1 = fn_fx1_pbe96(s1, kappa) ; fx1_2 = fn_fx1_pbe96(s2, kappa)
   fxp_1 = fn_fxp_pbe96(s1, kappa) ; fxp_2 = fn_fxp_pbe96(s2, kappa)
   fx_gga = rho * ex0 * 0.5d0 &
           * ( (1.d0+zeta)**(frac4_3) * fx0_1 &
              +(1.d0-zeta)**(frac4_3) * fx0_2 )
   ec_gga = ec_unif + h
   fc_gga = rho * ec_gga
   dfx_gga(1) = (frac4_3) * ex0_1 * (fx0_1 - fx1_1)
   dfx_gga(2) = (frac4_3) * ex0_2 * (fx0_2 - fx1_2)
   do ispin = 1,2
      z1 = + ec_gga
      z2 = - (frac1_3)*rs * (decdr_unif + dhdr)
      if ( 1.d0 - eps_zeta > abs(zeta) ) then
         z3 = - (zeta - dble(spin(ispin))) &
              * (decdz_unif + dhdz - dphi/phi*t*dhdt)
      else
         z3 = 0.d0
      end if
      z4 = - (frac7_6) * t * dhdt
      dfc_gga(ispin) = z1 + z2 + z3 + z4
   end do
   dfxda_gga(1) = -0.375d0 / pi * fxp_1
   dfxda_gga(2) = -0.375d0 / pi * fxp_2
   dfcda_gga = dhdt / (2.d0*phi*ks)
   end subroutine gga_pbe96_fxc

!=====================================================================
   subroutine gga_pbe96_spherical(imode, &
                 r,rho1,rho2,drho1_in,drho2_in,ddrho1_in,ddrho2_in, &
                 ex_gga,ec_gga,vx_gga,vc_gga)
!=====================================================================
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   integer,intent(in)  :: imode
   real(8),intent(in)  :: r, rho1, rho2
   real(8),intent(in)  :: drho1_in, drho2_in, ddrho1_in, ddrho2_in
   real(8),intent(out) :: ex_gga, ec_gga, vx_gga(2), vc_gga(2)
   integer :: ispin, spin(2)
   real(8) :: &
     drho1, drho2, ddrho1, ddrho2, &
     ex_unif, ec_unif, vx_unif(2), vc_unif(2), &
     decdr_unif, decdz_unif, &
     rho, drho, ddrho, rs, zeta, dzeta, phi, dphi, &
     ks, kf, kf1, kf2, s1, s2, t, &
     a, h, q0, q1, q2, u, exp_u, &
     dq0dz, dq1dr, dq1dz, dq1dt, dq2dr, dq2dz, dq2dt, &
     ddq1drdt, ddq1dzdt, ddq1dtdt, ddq2drdt, ddq2dzdt, ddq2dtdt, &
     dadr, dadz, dhdr, dhdz, dhdt, ddhdtdt, ddhdrdt, ddhdzdt, &
     t1dhdt, &
     fn_ex0, fn_fx_pbe96, fn_fx1_pbe96, fn_fx2_pbe96, &
     p0, att, t2, t3, t4, t5, t6, a2, qqq, &
     ex0, ex0_1, ex0_2, fx0_1, fx0_2, fx1_1, fx1_2, fx2_1, fx2_2, &
     z1, z2, z3, z4, z5, z6_z7, z8, sign_drho, &
     tmp1, tmp2
   real(8),parameter :: &
      argmax_exp = 300.d0,                               &
      argmin_ln  = 1.d-15,                               &
      pi      = 3.1415926535897932384626433832795029d0,  &
      pipi    = 9.8696044010893586188344909998761511d0,  &
      frac1_3 = 0.33333333333333333333333333333333333d0, &
      frac2_3 = 0.66666666666666666666666666666666666d0, &
      frac4_3 = 1.33333333333333333333333333333333333d0, &
      frac7_6 = 1.16666666666666666666666666666666666d0, &
      rhomin   = 1.d-25,                                 &
      eps_zeta = 1.d-5
   real(8) :: kappa, mu_kappa
   real(8),parameter :: &
      beta       = 0.06672455060314922d0, &
      mu         = 0.21951497278705521609828723131160858d0,  &
      gamma      = 0.031090690869654895034940863712730629d0, &
      beta_gamma = 2.1461263399673646060592325913558248d0
    drho1 = drho1_in  ;  drho2 = drho2_in
   ddrho1 = ddrho1_in ; ddrho2 = ddrho2_in
   select case (imode)
   case (0)
      kappa    = 0.8040d0
      mu_kappa = 0.27302857315259963733997193463227010d0
   case (1)
      kappa    = 1.2450d0
      mu_kappa = mu / kappa
   end select
   spin(1) = +1 ; spin(2) = -1     
   if (rho1 < rhomin) then
      drho1 = 0.d0 ; ddrho1 = 0.d0
   end if
   if (rho2 < rhomin) then
      drho2 = 0.d0 ; ddrho2 = 0.d0
   end if
   rho   = rho1   + rho2
   drho  = drho1  + drho2
   ddrho = ddrho1 + ddrho2
   if ((rho1 < rhomin).and.(rho2 < rhomin)) then
      zeta  = 0.d0
      dzeta = 0.d0
      phi   = 0.5d0
      dphi  = 0.d0
   else if ((rho1 >= rhomin).and.(rho2 < rhomin)) then
      zeta  = +1.d0
      dzeta =  0.d0
      phi   = 2.d0**(-frac1_3)
      dphi  = frac1_3 * 2.d0**(-frac1_3)
   else if ((rho1 < rhomin).and.(rho2 >= rhomin)) then
      zeta  = -1.d0
      dzeta =  0.d0
      phi   = 2.d0**(-frac1_3)
      dphi  = -frac1_3 * 2.d0**(-frac1_3)
   else
      zeta  = (rho1 - rho2) / rho
      dzeta = 2.d0 * (rho2*drho1 - rho1*drho2) / (rho*rho)
      phi   = 0.5d0 &
            * ((1.d0+zeta)**( frac2_3) + (1.d0-zeta)**( frac2_3))
      dphi  = (frac1_3) &
            * ((1.d0+zeta)**(-frac1_3) - (1.d0-zeta)**(-frac1_3))
   end if
   kf1   = (6.d0*pipi*rho1)**(frac1_3)
   kf2   = (6.d0*pipi*rho2)**(frac1_3)
   kf    = (3.d0*pipi*rho )**(frac1_3)
   ks    = sqrt(4.d0*kf/pi)
   rs    = (0.75d0/pi/rho)**(frac1_3)
   if (rho1 < rhomin) then
      s1 = 0.d0
   else
      s1 = abs(drho1) / (2.d0*kf1*rho1)
   end if
   if (rho2 < rhomin) then
      s2 = 0.d0
   else
      s2 = abs(drho2) / (2.d0*kf2*rho2)
   end if
   if (rho < rhomin) then
      t = 0.d0
   else
      t = abs(drho ) / (2.d0*phi*ks*rho)
   end if
   call lda_pw92(rho1,rho2,ex_unif,ec_unif,vx_unif,vc_unif, &
                 decdr_unif,decdz_unif)
   q0 = gamma * phi * phi * phi
      p0 = beta * phi * phi * phi
      t2 =  t * t ; t3 = t2 * t ; t4 = t3 * t
      t5 = t4 * t ; t6 = t5 * t
   u  = min(ec_unif/q0, -1.d-15)
      exp_u = exp(min(-u,argmax_exp))
   a  = beta_gamma / (exp_u - 1.d0)
      att = a * t2 ; a2 = a * a
   q1 = beta_gamma * t2 * (1.d0 + att)
   q2 = 1.d0 + att * (1.d0 + att)
   h  = q0 * log( max(1.d0+q1/q2, argmin_ln) )
   dadr = a2 * exp_u * decdr_unif/p0
   dadz = a2 * exp_u * 3.d0*ec_unif*dphi/(p0*phi)
   dq0dz = 3.d0 * gamma * phi * phi * dphi
   dq1dr = beta_gamma * t4 * dadr
   dq1dz = beta_gamma * t4 * dadz
   dq1dt = 2.d0 * beta_gamma * t * (1.d0 + 2.d0*att)
   dq2dr = t2 * (1.d0 + 2.d0*att) * dadr
   dq2dz = t2 * (1.d0 + 2.d0*att) * dadz
   dq2dt = 2.d0*a*t * (1.d0 + 2.d0*att)
   ddq1drdt = 4.d0 * beta_gamma * t3 * dadr
   ddq1dzdt = 4.d0 * beta_gamma * t3 * dadz
   ddq1dtdt = 2.d0 * beta_gamma * (1.d0 + 6.d0*att)
   ddq2drdt = 2.d0*t * (1.d0 + 4.d0*att) * dadr
   ddq2dzdt = 2.d0*t * (1.d0 + 4.d0*att) * dadz
   ddq2dtdt = 2.d0*a * (1.d0 + 6.d0*att)
      qqq = q0/(q1+q2)/q2
   dhdr = - qqq * beta_gamma *a*t6*(2.d0+att) * dadr 
   dhdz = dq0dz * (h/q0) - qqq * beta_gamma * a*t6*(2.d0+att) * dadz
   dhdt = qqq * 2.d0*beta_gamma * t*(1.d0+2.d0*att)
   t1dhdt = qqq * 2.d0*beta_gamma * (1.d0+2.d0*att)

   ddhdtdt = - dhdt * ( (dq1dt + dq2dt)/(q1 + q2) + dq2dt/q2 ) &
             + qqq * 2.d0*beta_gamma * (1.d0+6.d0*att)
   ddhdrdt = - dhdt * ( (dq1dr + dq2dr)/(q1 + q2) + dq2dr/q2 ) &
             + qqq * 4.d0*beta_gamma * t3 * dadr
   ddhdzdt = - dhdt * ( (dq1dz + dq2dz)/(q1 + q2) + dq2dz/q2   &
                                                  - dq0dz/q0 ) &
             + qqq * 4.d0*beta_gamma * t3 * dadz
   ex0   = fn_ex0(     rho )
   ex0_1 = fn_ex0(2.d0*rho1) ; ex0_2 = fn_ex0(2.d0*rho2)
   fx0_1 = fn_fx_pbe96 (s1, kappa) ; fx0_2 = fn_fx_pbe96 (s2, kappa)
   fx1_1 = fn_fx1_pbe96(s1, kappa) ; fx1_2 = fn_fx1_pbe96(s2, kappa)
   fx2_1 = fn_fx2_pbe96(s1, kappa) ; fx2_2 = fn_fx2_pbe96(s2, kappa)
   ex_gga = ex0 * 0.5d0 &
           * ( (1.d0+zeta)**(frac4_3) * fx0_1 &
              +(1.d0-zeta)**(frac4_3) * fx0_2 )
   ec_gga = ec_unif + h
   if (rho1 < rhomin) then
      tmp1 = 0.d0
   else
      tmp1 = (1.d0 - 3.d0*s1*s1*mu_kappa) / (1.d0 + s1*s1*mu_kappa) &
              * (ddrho1/rho1) &
           + 2.d0/r * (drho1/rho1)
      tmp1 = tmp1 &
              * 2.d0*mu * (0.5d0/kf1)**2 / (1.d0 + s1*s1*mu_kappa)**2
   end if
   if (rho2 < rhomin) then
      tmp2 = 0.d0
   else
      tmp2 = (1.d0 - 3.d0*s2*s2*mu_kappa) / (1.d0 + s2*s2*mu_kappa) &
              * (ddrho2/rho2) &
           + 2.d0/r * (drho2/rho2)
      tmp2 = tmp2 &
              * 2.d0*mu * (0.5d0/kf2)**2 / (1.d0 + s2*s2*mu_kappa)**2
   end if
   vx_gga(1) = ex0_1 * ( + (frac4_3) * (fx0_1 - fx2_1) - tmp1 )
   vx_gga(2) = ex0_2 * ( + (frac4_3) * (fx0_2 - fx2_2) - tmp2 )
   sign_drho = sign(1.d0,drho)
   do ispin = 1,2
      z1 = + ec_gga
      z2 = - (frac1_3)*rs * (decdr_unif + dhdr - t*ddhdrdt)
      if ( 1.d0 - eps_zeta > abs(zeta) ) then
         z3 = - (zeta - dble(spin(ispin))) &
              * (decdz_unif + dhdz - dphi/phi*t*dhdt)
      else
         z3 = 0.d0
      end if
      z4 = - t * dhdt
      z5 = + (frac7_6)*t2 * ddhdtdt
      z6_z7 = - (ddrho/rho * ddhdtdt + 2.d0/r * drho/rho * t1dhdt) &
                  / (2.d0*phi*ks)**2
      if ( 1.d0 - eps_zeta > abs(zeta) ) then
         z8 = + sign_drho * dzeta / (2.d0*phi*ks) &
              * (dphi/phi * (dhdt + t*ddhdtdt) - ddhdzdt)
      else
         z8 = 0.d0
      end if
      vc_gga(ispin) = z1 + z2 + z3 + z4 + z5 + z6_z7 + z8
   end do
   end subroutine gga_pbe96_spherical

!=====================================================================
   subroutine gga_pw91_fxc( &
                 rho1_in,rho2_in,drho1_in,drho2_in,drho_in, &
                 fx_gga,fc_gga,dfx_gga,dfc_gga,dfxda_gga,dfcda_gga)
!=====================================================================
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   real(8),intent(in)  :: rho1_in, rho2_in, &
                          drho1_in, drho2_in, drho_in
   real(8),intent(out) :: fx_gga, fc_gga, &
                          dfx_gga(2), dfc_gga(2), &
                          dfxda_gga(2), dfcda_gga
   integer :: ispin, spin(2)
   real(8) :: &
     rho1, rho2, drho1, drho2, &
     ex_unif, ec_unif, vx_unif(2), vc_unif(2), &
     decdr_unif, decdz_unif, &
     rho, drho, rs, zeta, dzeta, phi, dphi, &
     ks, kf, kf1, kf2, s1, s2, t, &
     a, h, q0, q1, q2, u, exp_u, &
     dq0dz, dq1dr, dq1dz, dq1dt, dq2dr, dq2dz, dq2dt, &
     dadr, dadz, dhdr, dhdz, dhdt, &
     p0, att, t2, t3, t4, t5, t6, a2, qqq, &
     ex0, ex0_1, ex0_2, fx0_1, fx0_2, fxp_1, fxp_2, &
     z1, z2, z3, z4, ec_gga, &
     c, dc, r0, r1, j, jexp, djdr, djdz, djdt, &
     fn_ex0, fn_fx_pw91, fn_fxp_pw91, fn_c_pw91, fn_cp_pw91
   real(8),parameter :: &
      argmax_exp = 300.d0,                               &
      argmin_ln  = 1.d-15,                               &
      pi      = 3.1415926535897932384626433832795029d0,  &
      pipi    = 9.8696044010893586188344909998761511d0,  &
      frac1_3 = 0.33333333333333333333333333333333333d0, &
      frac2_3 = 0.66666666666666666666666666666666666d0, &
      frac4_3 = 1.33333333333333333333333333333333333d0, &
      frac7_6 = 1.16666666666666666666666666666666666d0, &
      rhomin   = 1.d-25,                                 &
      eps_zeta = 1.d-5
   real(8),parameter :: &
      beta       =  0.0667263d0, &
      gamma      =  0.0247356d0, &
      beta_gamma =  2.69759d0,   &
      nu         = 15.75592034948314465886357d0, &
      mu         = 66.34364396064500703774351d0
    rho1  =  rho1_in ; rho2  =  rho2_in
    drho1 = drho1_in ; drho2 = drho2_in ; drho = drho_in
   spin(1) = +1 ; spin(2) = -1     
   rho = rho1 + rho2
   if (rho1 < rhomin) then
      drho1 = 0.d0
   end if
   if (rho2 < rhomin) then
      drho2 = 0.d0
   end if
   if (rho < rhomin) then
      drho = 0.d0
   end if
   if ((rho1 < rhomin).and.(rho2 < rhomin)) then
      zeta  = 0.d0
      dzeta = 0.d0
      phi   = 0.5d0
      dphi  = 0.d0
   else if ((rho1 >= rhomin).and.(rho2 < rhomin)) then
      zeta  = +1.d0
      dzeta =  0.d0
      phi   = 2.d0**(-frac1_3)
      dphi  = frac1_3 * 2.d0**(-frac1_3)
   else if ((rho1 < rhomin).and.(rho2 >= rhomin)) then
      zeta  = -1.d0
      dzeta =  0.d0
      phi   = 2.d0**(-frac1_3)
      dphi  = -frac1_3 * 2.d0**(-frac1_3)
   else
      zeta  = (rho1 - rho2) / rho
      dzeta = 2.d0 * (rho2*drho1 - rho1*drho2) / (rho*rho)
      phi   = 0.5d0 &
            * ((1.d0+zeta)**( frac2_3) + (1.d0-zeta)**( frac2_3))
      dphi  = (frac1_3) &
            * ((1.d0+zeta)**(-frac1_3) - (1.d0-zeta)**(-frac1_3))
   end if
   kf1   = (6.d0*pipi*rho1)**(frac1_3)
   kf2   = (6.d0*pipi*rho2)**(frac1_3)
   kf    = (3.d0*pipi*rho )**(frac1_3)
   ks    = sqrt(4.d0*kf/pi)
   rs    = (0.75d0/pi/rho)**(frac1_3)
   if (rho1 < rhomin) then
      s1 = 0.d0
   else
      s1 = abs(drho1) / (2.d0*kf1*rho1)
   end if
   if (rho2 < rhomin) then
      s2 = 0.d0
   else
      s2 = abs(drho2) / (2.d0*kf2*rho2)
   end if
   if (rho < rhomin) then
      t = 0.d0
   else
      t = abs(drho) / (2.d0*phi*ks*rho)
   end if
   call lda_pw92(rho1,rho2,ex_unif,ec_unif,vx_unif,vc_unif, &
                 decdr_unif,decdz_unif)
   q0 = gamma * phi * phi * phi
      p0 = beta * phi * phi * phi
      t2 =  t * t ; t3 = t2 * t ; t4 = t3 * t
      t5 = t4 * t ; t6 = t5 * t
   u  = min(ec_unif/q0, -1.d-15)
      exp_u = exp(min(-u,argmax_exp))
   a  = beta_gamma / (exp_u - 1.d0)
      att = a * t2 ; a2 = a * a
   q1 = beta_gamma * t2 * (1.d0 + att)
   q2 = 1.d0 + att * (1.d0 + att)
   h  = q0 * log( max(1.d0+q1/q2, argmin_ln) )
   dadr = a2 * exp_u * decdr_unif/p0
   dadz = a2 * exp_u * 3.d0*ec_unif*dphi/(p0*phi)
   dq0dz = 3.d0 * gamma * phi * phi * dphi
   dq1dr = beta_gamma * t4 * dadr
   dq1dz = beta_gamma * t4 * dadz
   dq1dt = 2.d0 * beta_gamma * t * (1.d0 + 2.d0*att)
   dq2dr = t2 * (1.d0 + 2.d0*att) * dadr
   dq2dz = t2 * (1.d0 + 2.d0*att) * dadz
   dq2dt = 2.d0*a*t * (1.d0 + 2.d0*att)
      qqq = q0/(q1+q2)/q2
   dhdr = - qqq * beta_gamma *a*t6*(2.d0+att) * dadr 
   dhdz = dq0dz * (h/q0) - qqq * beta_gamma * a*t6*(2.d0+att) * dadz
   dhdt = qqq * 2.d0*beta_gamma * t*(1.d0+2.d0*att)
   c  = fn_c_pw91(rs)
   dc = fn_cp_pw91(rs)
   r0 = nu * c * phi**3 * t**2
   r1 = mu * rs * phi**4 * t**2
   if (r1 < argmax_exp) then
      jexp = exp(-r1)
   else
      jexp = 0.d0
   end if
   j = r0 * jexp
   djdr = jexp * (nu*phi**3*t**2) * (dc - c*mu*phi**4*t**2)
   djdz = jexp * (nu*c*dphi*phi**2*t**2) * (3.d0 - 4.d0*r1)
   djdt = jexp * (nu*c*phi**3*t) * (2.d0 - 2.d0*r1)
   ex0   = fn_ex0(     rho )
   ex0_1 = fn_ex0(2.d0*rho1) ; ex0_2 = fn_ex0(2.d0*rho2)
   fx0_1 = fn_fx_pw91 (s1)   ; fx0_2 = fn_fx_pw91 (s2)
   fxp_1 = fn_fxp_pw91(s1)   ; fxp_2 = fn_fxp_pw91(s2)
   fx_gga = rho * ex0 * 0.5d0 &
           * ( (1.d0+zeta)**(frac4_3) * fx0_1 &
              +(1.d0-zeta)**(frac4_3) * fx0_2 )
   ec_gga = ec_unif + h + j
   fc_gga = rho * ec_gga
   dfx_gga(1) = (frac4_3) * ex0_1 * (fx0_1 - s1*fxp_1)
   dfx_gga(2) = (frac4_3) * ex0_2 * (fx0_2 - s2*fxp_2)
   do ispin = 1,2
      z1 = + ec_gga
      z2 = - (frac1_3)*rs * (decdr_unif + dhdr + djdr)
      if ( 1.d0 - eps_zeta > abs(zeta) ) then
         z3 = - (zeta - dble(spin(ispin))) &
              * (decdz_unif + dhdz + djdz - dphi/phi*t*(dhdt + djdt))
      else
         z3 = 0.d0
      end if
      z4 = - (frac7_6) * t * (dhdt + djdt)
      dfc_gga(ispin) = z1 + z2 + z3 + z4
   end do
   dfxda_gga(1) = -0.375d0 / pi * fxp_1
   dfxda_gga(2) = -0.375d0 / pi * fxp_2
   dfcda_gga = (dhdt + djdt) / (2.d0*phi*ks)
   end subroutine gga_pw91_fxc

!=====================================================================
   function fn_ex0(rho)
!=====================================================================  
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   real(8)            :: fn_ex0
   real(8),intent(in) :: rho
   real(8),parameter  :: &
      pi      = 3.1415926535897932384626433832795029d0, &
      frac1_3 = 0.33333333333333333333333333333333333d0, &
      aa = 0.23873241463784300365332564505877154d0, &
      bb = 3.09366772628013593096894520722238695d0, &
      cc = 0.73855876638202240588423003268083626d0
   fn_ex0 = - cc * (rho)**(frac1_3)
   end function fn_ex0

!=====================================================================
   function fn_fx_pbe96(s,kappa)
!=====================================================================  
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   real(8)            :: fn_fx_pbe96
   real(8),intent(in) :: s, kappa
   real(8)            :: s2, mus2, mus2k
   real(8),parameter  :: &
      pi    = 3.1415926535897932384626433832795029d0, &
      beta  = 0.06672455060314922d0, &
      mu    = 0.21951497278705521609828723131160858d0
   s2    = s*s
   mus2  = mu*s2
   mus2k = mus2/kappa
   if (mus2k < 1.d0) then
      fn_fx_pbe96 = 1.d0 + mus2/(1.d0 + mus2k)
   else
      fn_fx_pbe96 = 1.d0 + kappa - kappa/(1.d0 + mus2k)
   end if
   end function fn_fx_pbe96

!=====================================================================
   function fn_fx1_pbe96(s,kappa)
!=====================================================================  
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   real(8)            :: fn_fx1_pbe96
   real(8),intent(in) :: s, kappa
   real(8),parameter  :: &
      pi    = 3.1415926535897932384626433832795029d0, &
      beta  = 0.06672455060314922d0, &
      mu    = 0.21951497278705521609828723131160858d0
   fn_fx1_pbe96 = 2.d0*mu*s*s / (1.d0 + mu*s*s/kappa)**2.d0
   end function fn_fx1_pbe96

!=====================================================================
   function fn_fxp_pbe96(s,kappa)
!=====================================================================  
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   real(8)            :: fn_fxp_pbe96
   real(8),intent(in) :: s, kappa
   real(8),parameter  :: &
      pi    = 3.1415926535897932384626433832795029d0, &
      beta  = 0.06672455060314922d0, &
      mu    = 0.21951497278705521609828723131160858d0
   fn_fxp_pbe96 = 2.d0*mu*s / (1.d0 + mu*s*s/kappa)**2.d0
   end function fn_fxp_pbe96

!=====================================================================
   function fn_fx2_pbe96(s,kappa)
!=====================================================================  
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   real(8)            :: fn_fx2_pbe96
   real(8),intent(in) :: s, kappa
   real(8),parameter  :: &
      pi    = 3.1415926535897932384626433832795029d0, &
      beta  = 0.06672455060314922d0, &
      mu    = 0.21951497278705521609828723131160858d0
   fn_fx2_pbe96 = &
      8.d0*(mu*s*s)**2.d0 / kappa / (1.d0 + mu*s*s/kappa)**3.d0
   end function fn_fx2_pbe96

!=====================================================================
   function fn_fx_pw91(s_in)
!=====================================================================  
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   real(8)            :: fn_fx_pw91
   real(8),intent(in) :: s_in
   real(8)            :: s, s2, ash, ees, llsash, aae, arg
   real(8),parameter  :: &
      argmax = 300.d0, &
      ll =  0.19645d0, &
      ee =  7.7956d0,  &
      a0 =  0.2743d0,  &
      a1 = -0.1508d0,  &
      b  =  0.004d0
   s      = abs(s_in)
   s2     = s*s
   ees    = ee*s
   ash    = log(ees + sqrt(1.d0 + ees*ees))
   llsash = ll*s*ash
   arg = 100.d0*s2
   aae = a0
   if (arg < argmax) then
      aae = aae + a1*exp(-arg)
   end if
   fn_fx_pw91 = (1.d0 + llsash + aae*s2) / (1.d0 + llsash + b*s2*s2)
   end function fn_fx_pw91

!=====================================================================
   function fn_fxp_pw91(s_in)
!=====================================================================  
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   real(8)            :: fn_fxp_pw91
   real(8),intent(in) :: s_in
   real(8)            :: numer1, numer2, denom, s, s2, ash, ees, sq, &
                         llterm, llsash, aae, ase, arg, exa
   real(8),parameter  :: &
      argmax = 300.d0, ll =  0.19645d0, ee =  7.7956d0, &
      a0 =  0.2743d0,  a1 = -0.1508d0,  b  =  0.004d0
   s      = abs(s_in)
   s2     = s*s
   ees    = ee*s
   sq     = sqrt(1.d0 + ees*ees)
   ash    = log(ees + sq)
   llsash = ll*s*ash
   llterm = ll*ees/sq + ll*ash
   arg = 100.d0*s2
   aae = a0 ; ase = 0.d0
   if (arg < argmax) then
      exa = exp(-arg)
      aae = aae + a1*exa
      ase = 200.d0*a1*s2*s*exa
   end if
   numer1 = - (4.d0*b*s2*s + llterm) * (1.d0 + llsash + aae*s2)
   denom  = 1.d0 + llsash + b*s2*s2
   numer2 = (2.d0*s*aae - ase + llterm) * denom
   denom  = denom*denom
   fn_fxp_pw91 = (numer1 + numer2) / denom
   end function fn_fxp_pw91

!=====================================================================
   function fn_c_pw91(rs)
!=====================================================================  
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   real(8)            :: fn_c_pw91
   real(8),intent(in) :: rs
   real(8)            :: numer, denom, cxc, cxc0
   real(8),parameter  :: &
      frac3_7 = 0.428571428571428571428571428571d0, &
      cx = -0.001667d0, &
      c0 =  2.568d0, c1 = 23.266d0, c2 =  7.389d-3, &
      d1 =  8.723d0, d2 =  0.472d0, d3 =  7.389d-2
   numer = c0 + rs*(c1 + c2*rs)
   denom = 1.d0 + rs*(d1 + rs*(d2 + d3*rs))
   cxc   = numer / denom * 1.d-3
   cxc0  = c0 * 1.d-3
   fn_c_pw91 = cxc - cxc0 - frac3_7*cx
   end function fn_c_pw91

!=====================================================================
   function fn_cp_pw91(rs)
!=====================================================================  
!
!  M. Okamoto
!
!---------------------------------------------------------------------
   implicit none
   real(8)            :: fn_cp_pw91
   real(8),intent(in) :: rs
   real(8)            :: denom, t0, t1, t2
   real(8),parameter  :: &
      c0 =  2.568d0, c1 = 23.266d0, c2 =  7.389d-3, &
      d1 =  8.723d0, d2 =  0.472d0, d3 =  7.389d-2
   t0 = - c0 * (d1 + rs*(2.d0*d2 + 3.d0*d3*rs))
   t1 = + c1 * (1.d0 - rs*rs*(d2 + 2.d0*d3*rs))
   t2 = + c2 * rs * (2.d0 + rs*(d1 - d3*rs*rs))
   denom = 1.d0 + rs*(d1 + rs*(d2 + d3*rs))
   denom = denom * denom
   fn_cp_pw91 = (t0 + t1 + t2) / denom * 1.d-3
   end function fn_cp_pw91
