! @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ !
! @@                                                                @@ !
! @@       PROGRAM  ASCOT 2014.440 (ver.4.53)                       @@ !
! @@     "Abinitio Simulation Code for Quantum Transport"           @@ !
! @@                                                                @@ !
! @@                                                                @@ !
! @@  AUTHOR(S): Naoki WATANABE, Nobutaka NISHIKAWA (Mizuho I.R.)   @@ !
! @@             Hisashi KONDO (Univ. Tokyo)                        @@ !
! @@                                                09/May/2014     @@ !
! @@                                                                @@ !
! @@  Contact address: Phase System Consortium                      @@ !
! @@                                                                @@ !
! @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ !

module ac_sharmonics

  integer, parameter :: N = 64  
  real(8), pointer ::  vshx(:)  
  real(8), pointer ::  vshw(:)  

  real(8), parameter :: C00 = 0.282094791773878d0 
  real(8), parameter :: C10 = 0.488602511902920d0 
  real(8), parameter :: C11 = 0.488602511902920d0 
  real(8), parameter :: C20 = 0.315391565252520d0 
  real(8), parameter :: C21 = 1.092548430592079d0 
  real(8), parameter :: C22 = 0.546274215296040d0 
  real(8), parameter :: C30 = 0.373176332590115d0 
  real(8), parameter :: C31 = 0.457045799464466d0 
  real(8), parameter :: C32 = 1.445305721320277d0 
  real(8), parameter :: C33 = 0.590043589926644d0 
  real(8), parameter :: C40 = 0.105785546915204d0 
  real(8), parameter :: C41 = 0.669046543557289d0 
  real(8), parameter :: C42 = 0.473087347878780d0 
  real(8), parameter :: C43 = 1.770130769779931d0 
  real(8), parameter :: C44 = 0.625835735449176d0 
  real(8), parameter :: C50 = 0.116950322453424d0 
  real(8), parameter :: C51 = 0.452946651195697d0 
  real(8), parameter :: C52 = 2.396768392486662d0 
  real(8), parameter :: C53 = 0.489238299435250d0 
  real(8), parameter :: C54 = 2.075662314881042d0 
  real(8), parameter :: C55 = 0.656382056840170d0 
  real(8), parameter :: C60 = 0.063569202267628d0 
  real(8), parameter :: C61 = 0.582621362518731d0 
  real(8), parameter :: C62 = 0.460602629757462d0 
  real(8), parameter :: C63 = 0.921205259514924d0 
  real(8), parameter :: C64 = 0.504564900728724d0 
  real(8), parameter :: C65 = 2.366619162231752d0 
  real(8), parameter :: C66 = 0.683184105191914d0 

end module ac_sharmonics

subroutine SphericalHarmonic__rlY( l, m, R, rlY )
  use ac_sharmonics

  implicit none
  integer, intent(in) :: l, m
  real(8), intent(in) :: R(3)
  real(8)  :: rlY
  real(8)  :: x, y, z, dR2

  x = R(1)
  y = R(2)
  z = R(3)
  dR2 = dot_product(R,R)
  rlY = 0.d0

  select case(l)
  case(0) 
     rlY = C00 

  case(1) 
     select case(m)
     case(+1)
        rlY = C11*x 
     case(-1)
        rlY = C11*y 
     case( 0)
        rlY = C10*z 
     end select

  case(2) 
     select case(m)
     case( 0)
        rlY = C20*(3*z*z-dR2) 
     case(+2)
        rlY = C22*(x*x-y*y) 
     case(-2)
        rlY = C22*2.0*x*y 
     case(+1)
        rlY = C21*z*x 
     case(-1)
        rlY = C21*y*z 
     end select

  case(3) 
     select case(m)
     case( 0)
        rlY = C30*z * (5*z*z-3*dR2)
     case(+1)
        rlY = C31*x * (5*z*z-dR2)
     case(-1)
        rlY = C31*y * (5*z*z-dR2)
     case(+2)
        rlY = C32*z * (x*x-y*y)
     case(-2)
        rlY = C32*z * (2.0*x*y)
     case(+3)
        rlY = C33*x * (x*x - 3*y*y)
     case(-3)
        rlY = C33*y * (3*x*x - y*y)
     end select


  case(4) 
     select case(m)
     case(0)
        rlY = C40*(35*z*z*z*z - 30*z*z*dR2 + 3*dR2*dR2 )
     case(+1)
        rlY = C41*x*z * (7*z*z-3*dR2)

     case(-1)
        rlY = C41*y*z * (7*z*z-3*dR2)

     case(+2)
        rlY = C42*(x*x-y*y) * (7*z*z-dR2)

     case(-2)
        rlY = C42*2*x*y * (7*z*z-dR2)

     case(+3)
        rlY = C43*x*z * (x*x-3*y*y)

     case(-3)
        rlY = C43*y*z * (3*x*x-y*y)

     case(+4)
        rlY = C44 * (x*x-2*x*y-y*y) * &
             (x*x+2*x*y-y*y)
     case(-4)
        rlY = C44 * (4*x*y) * (x*x-y*y)

     end select


  case(5) 
     select case(m)
     case(0)
        rlY = C50*z* (63*z*z*z*z-70*z*z*dR2+15*dR2*dR2)
     case(+1)
        rlY = C51*x* (21*z*z*z*z-14*z*z*dR2+dR2*dR2 )
     case(-1)
        rlY = C51*y* (21*z*z*z*z-14*z*z*dR2+dR2*dR2 )
     case(+2)
        rlY = C52*z* (x*x-y*y) * (3*z*z-dR2)
     case(-2)
        rlY = C52*z* (2*x*y) * (3*z*z-dR2)
     case(+3)
        rlY = C53*x* (x*x-3*y*y) * (9*z*z-dR2)
     case(-3)
        rlY = C53*y* (3*x*x-y*y) * (9*z*z-dR2)
     case(+4)
        rlY = C54*z* (x*x-2*x*y-y*y) * &
             (x*x+2*x*y-y*y)
     case(-4)
        rlY = C54*z* (4*x*y) * (x*x-y*y)
     case(+5)
        rlY = C55*x* (x*x*x*x-10*x*x*y*y+5*y*y*y*y)
     case(-5)
        rlY = C55*y* (5*x*x*x*x-10*x*x*y*y+y*y*y*y)
     end select

  case(6) 
     select case(m)
     case(0)
        rlY = C60*( 231*z*z*z*z*z*z &
             - 315*z*z*z*z*dR2 + 105*z*z*dR2*dR2 &
             - 5*dR2*dR2*dR2 )
     case(+1)
        rlY = C61*z*x * &
             (33*z*z*z*z-30*z*z*dR2+5*dR2*dR2)
     case(-1)
        rlY = C61*z*y * &
             (33*z*z*z*z-30*z*z*dR2+5*dR2*dR2)
     case(+2)
        rlY = C62*(x*x-y*y) * &
             (33*z*z*z*z-18*z*z*dR2+dR2*dR2)
     case(-2)
        rlY = C62*(2*x*y)* &
             (33*z*z*z*z-18*z*z*dR2+dR2*dR2)
     case(+3)
        rlY = C63*z*x * &
             (x*x-3*y*y) * (11*z*z-3*dR2)
     case(-3)
        rlY = C63*z*y * &
             (3*x*x-y*y) * (11*z*z-3*dR2)
     case(+4)
        rlY = C64*(11*z*z-dR2) * &
             (x*x-2*x*y-y*y) * (x*x+2*x*y-y*y)
     case(-4)
        rlY = C64*(11*z*z-dR2) * &
             4*x*y * (x*x-y*y)
     case(+5)
        rlY = C65*z*x * &
             (x*x*x*x-10*x*x*y*y+5*y*y*y*y)
     case(-5)
        rlY = C65*z*y * &
             (5*x*x*x*x-10*x*x*y*y+y*y*y*y)
     case(+6)
        rlY = C66*(x*x-y*y) * &
             (x*x-4*x*y+y*y) * (x*x+4*x*y+y*y)
     case(-6)
        rlY = C66*2*x*y * &
             (3*x*x-y*y) * (x*x-3*y*y)
     end select

  case default
     rlY = 0.d0
     write(*,'(a70)') &
          '# Sorry : Spherical Harmonic function for l>6 is not implemented.'
     stop

  end select

  return
end subroutine SphericalHarmonic__rlY

subroutine SphericalHarmonic__drlY( l, m, R, drlY )
  use ac_sharmonics

  implicit none
  integer, intent(in) :: l, m
  real(8), intent(in) :: R(3)
  real(8)  :: drlY(3)

  real(8)  :: x, y, z, dR2, dR4

  x = R(1)
  y = R(2)
  z = R(3)
  drlY = (/ 0.d0, 0.d0, 0.d0 /)
  dR2  = dot_product(R,R)

  select case(l)
  case(0) 
     drlY = (/ 0.d0, 0.d0, 0.d0 /)

  case(1) 
     select case(m)
     case( 0)
        drlY = (/ 0.d0, 0.d0, C10 /)
     case(+1)
        drlY = (/ C11, 0.d0, 0.d0 /)
     case(-1)
        drlY = (/ 0.d0, C11, 0.d0 /)
     end select

  case(2) 
     select case(m)
     case( 0)
        drlY = C20*(/ -2*x, -2*y, 4*z /) 
     case(+1)
        drlY = C21*(/    z,   0.d0,   x /) 
     case(-1)
        drlY = C21*(/   0.d0,    z,   y /) 
     case(+2)
        drlY = C22*(/  2*x,- 2*y,  0.d0 /) 
     case(-2)
        drlY = C22*(/  2*y,  2*x,  0.d0 /) 
     end select

  case(3) 
     select case(m)
     case( 0)
        drlY = C30*(/ -6*x*z, -6*y*z, 9*z*z-3*dR2 /) 
     case(+1)
        drlY = C31*(/ -2*x*x + 5*z*z - dR2, -2*x*y, 8*x*z /)
     case(-1)
        drlY = C31*(/ -2*x*y,  -2*y*y + 5*z*z - dR2, 8*y*z /)
     case(+2)
        drlY = C32*(/ 2*x*z      , -2*y*z     , x*x-y*y /)
     case(-2)
        drlY = C32*(/ 2*y*z      ,  2*z*x     , 2*x*y /)
     case(+3)
        drlY = C33*(/ 3*(x*x-y*y), -6*x*y     , 0.d0 /)
     case(-3)
        drlY = C33*(/ 6*x*y      , 3*(x*x-y*y), 0.d0 /)
     end select

  case(4) 
     select case(m)
     case( 0)
        drlY = C40*(/ &
             -60*z*z*x + 12*x*dR2, &
             -60*z*z*y + 12*y*dR2, &
             80*z*z*z - 48*z*dR2 /)
     case(+1)
        drlY = C41*(/ &
             z*(7*z*z-3*dR2) - 6*x*x*z, &
             -6*x*y*z, &
             x*(7*z*z-3*dR2) + 8*x*z*z /)
     case(-1)
        drlY = C41*(/ &
             -6*x*y*z, &
             z*( 4*z*z - 3*x*x - 9*y*y), &
             y*(15*z*z-3*dR2)  /)
     case(+2)
        drlY = C42*(/ &
             4*x*(3*z*z - x*x), &
             4*y*( y*y - 3*z*z), &
             12*z*(x*x-y*y) /)
     case(-2)
        drlY = C42*(/ &
             2*y*(6*z*z - 3*x*x - y*y), &
             2*x*(6*z*z -   x*x - 3*y*y), &
             24*x*y*z /)
     case(+3)
        drlY = C43*(/ &
             3*z*(x*x-y*y), &
             -6*x*y*z, &
             x*(x*x-3*y*y) /)
     case(-3)
        drlY = C43*(/ &
             6*x*y*z, &
             3*z*(x*x-y*y), &
             y*(3*x*x-y*y) /)
     case(+4)
        drlY = C44*(/ &
             4*x*(x*x-3*y*y), &
             4*y*(y*y-3*x*x), &
             0.d0 /)
     case(-4)
        drlY = C44*(/ &
             4*y*(3*x*x - y*y), &
             4*x*(x*x - 3*y*y), &
             0.d0 /)
     end select

  case(5) 
     select case(m)
     case( 0)
        drlY = C50*(/ &
             z*(-140*x*z*z + 60*x*dR2), &
             z*(-140*y*z*z + 60*y*dR2), &
             175*z*z*z*z - 150*z*z*dR2 + 15*dR2*dR2 /)
     case(+1)
        drlY = C51*(/ &
             21*z*z*z*z - 14*z*z*dR2 + dR2*dR2 + 4*x*x*(dR2-7*z*z), &
             4*x*y*(dR2-7*z*z), &
             8*x*z*(7*z*z - 3*dR2) /)
     case(-1)
        drlY = C51*(/ &
             4*x*y*(dR2-7*z*z), &
             21*z*z*z*z - 14*z*z*dR2 + dR2*dR2 + 4*y*y*(dR2-7*z*z), &
             8*y*z*(7*z*z - 3*dR2) /)
     case(+2)
        drlY = C52*(/ &
             4*x*z*(z*z-x*x), &
             4*y*z*(y*y-z*z), &
             (7*z*z-dR2)*(x*x-y*y) /)
     case(-2)
        drlY = C52*(/ &
             2*y*z*(2*z*z - 3*x*x - y*y), &
             2*z*x*(2*z*z - 3*y*y - x*x), &
             2*x*y*(6*z*z - x*x - y*y) /)
     case(+3)
        drlY = C53*(/ &
             3*(x*x-y*y)*(9*z*z-dR2) - 2*x*x*(x*x-3*y*y), &
             - 4*x*y*(12*z*z-x*x-3*y*y), &
             + 16*x*z*(x*x-3*y*y) /)
     case(-3)
        drlY = C53*(/ &
             4*x*y*(12*z*z-3*x*x-y*y), &
             3*(x*x-y*y)*(9*z*z-dR2) - 2*y*y*(3*x*x-y*y), &
             16*y*z*(3*x*x-y*y) /)
     case(+4)
        drlY = C54*(/ &
             4*x*z*(x*x-3*y*y), &
             4*y*z*(y*y-3*x*x), &
             (x*x-2*x*y-y*y)*(x*x+2*x*y-y*y) /)
     case(-4)
        drlY = C54*(/ &
             4*y*z*(3*x*x-y*y), &
             4*z*x*(x*x-3*y*y), &
             4*x*y*(x*x-y*y) /)
     case(+5)
        drlY = C55*(/ &
             5*(x*x-2*x*y-y*y)*(x*x+2*x*y-y*y), &
             -20*x*y*(x*x-y*y), &
             0.d0 /)
     case(-5)
        drlY = C55*(/ &
             20*x*y*(x*x-y*y), &
             5*(x*x-2*x*y-y*y)*(x*x+2*x*y-y*y), &
             0.d0 /)
     end select

  case(6) 
     dR4 = dR2*dR2
     select case(m)
     case( 0)
        drlY = C60*(/ &
             -630*x*z*z*z*z + 420*x*z*z*dR2 - 30*x*dR4, &
             -630*y*z*z*z*z + 420*y*z*z*dR2 - 30*y*dR4, &
             756*z*z*z*z*z - 840*z*z*z*dR2 +180*z*dR4 /)
     case(+1)
        drlY = C61*(/ &
             + z*(33*z*z*z*z-30*z*z*dR2+5*dR4) - 20*z*x*x*(3*z*z-dR2), &
             -20*x*y*z*(3*z*z-dR2), &
             x*(105*z*z*z*z-70*z*z*dR2+5*dR4) /)
     case(-1)
        drlY = C61*(/ &
             -20*x*y*z*(3*z*z-dR2), &
             + z*(33*z*z*z*z-30*z*z*dR2+5*dR4) - 20*z*y*y*(3*z*z-dR2), &
             + y*(33*z*z*z*z-30*z*z*dR2+5*dR4)  + 8*z*z*y*(9*z*z-5*dR2) /)
     case(+2)
        drlY = C62*(/ &
             + 2*x*(33*z*z*z*z-18*z*z*dR2+dR4) - 4*x*(x*x-y*y)*(9*z*z-dR2), &
             - 2*y*(33*z*z*z*z-18*z*z*dR2+dR4) - 4*y*(x*x-y*y)*(9*z*z-dR2), &
             32*z*(x*x-y*y)*(3*z*z-dR2) /)
     case(-2) 
        drlY = C62*(/ &
             + 2*y*(33*z*z*z*z-18*z*z*dR2+dR4) - 8*x*x*y*(9*z*z-dR2), &
             + 2*x*(33*z*z*z*z-18*z*z*dR2+dR4) - 8*x*y*y*(9*z*z-dR2), &
             64*x*y*z*(3*z*z-dR2) /)
     case(+3)
        drlY = C63*(/ &
             3*(x*x-y*y)*z*(11*z*z-3*dR2) - 6*x*x*z*(x*x-3*y*y), &
             - 6*x*y*z*((11*z*z-3*dR2)+(x*x-3*y*y)), &
             x*(x*x-3*y*y)*(11*z*z-3*dR2) + 16*x*z*z*(x*x-3*y*y) /)
     case(-3)
        drlY = C63*(/ &
             12*x*y*z*(4*z*z-3*x*x-y*y), &
             3*(x*x-y*y)*z*(11*z*z-3*dR2) - 6*y*y*z*(3*x*x-y*y), &
             y*(3*x*x-y*y)*(11*z*z-3*dR2) + 16*y*z*z*(3*x*x-y*y) /)
     case(+4)
        drlY = C64*(/ &
             2*x*( 20*x*x*z*z - 60*y*y*z*z -3*x*x*x*x + 10*x*x*y*y + 5*y*y*y*y), &
             2*y*( 20*y*y*z*z - 60*x*x*z*z -3*y*y*y*y + 10*x*x*y*y + 5*x*x*x*x), &
             20*z*(x*x-2*x*y-y*y)*(x*x+2*x*y-y*y) /)
     case(-4)
        drlY = C64*(/ &
             + 4*y*( 30*x*x*z*z - 10*y*y*z*z - 5*x*x*x*x + y*y*y*y), &
             - 4*x*( 30*y*y*z*z - 10*x*x*z*z - 5*y*y*y*y + x*x*x*x), &
             80*x*y*z*(x*x-y*y) /)
     case(+5)
        drlY = C65*(/ &
             5*z*(x*x-2*x*y-y*y)*(x*x+2*x*y-y*y), &
             -20*x*y*z*(x*x-y*y), &
             x*(x*x*x*x-10*x*x*y*y+5*y*y*y*y) /)
     case(-5)
        drlY = C65*(/ &
             20*x*y*z*(x*x-y*y), &
             5*z*(x*x-2*x*y-y*y)*(x*x+2*x*y-y*y), &
             y*(5*x*x*x*x-10*x*x*y*y+y*y*y*y) /)
     case(+6)
        drlY = C66*(/ &
             +6*x*(5*y*y*y*y-10*x*x*y*y+x*x*x*x), &
             -6*y*(5*x*x*x*x-10*x*x*y*y+y*y*y*y), &
             0.d0 /)
     case(-6)
        drlY = C66*(/ &
             +6*y*(y*y*y*y-10*x*x*y*y+5*x*x*x*x), &
             +6*x*(x*x*x*x-10*x*x*y*y+5*y*y*y*y), &
             0.d0 /)
     end select

  case default
     drlY = (/ 0.d0, 0.d0, 0.d0 /) 
     write(*,'(a70)') &
          '# Sorry : Spherical Harmonic function for l>6 is not implemented.'
     stop

  end select

  return
end subroutine SphericalHarmonic__drlY

subroutine SphericalHarmonic__Gaunt( la, ma, lb, mb, l, m, g )
  use ac_sharmonics
  use ac_parameter

  implicit none
  integer, intent(in) :: l, la, lb
  integer, intent(in) :: m, ma, mb
  real(8) :: g

  real(8) :: tsum, psum
  integer :: it, ip
  real(8) :: th, ph, weight

  real(8) SphericalHarmonic__Theta
  real(8) SphericalHarmonic__Phi

  if( abs(abs(ma)-abs(mb)) /= abs(m) .and. abs(ma)+abs(mb) /= abs(m) ) then
     g = 0.d0
     return
  end if

  if( abs(mod(la-lb-l,4)) /= 0 .and. abs(mod(la-lb-l,4)) /= 2 ) then
     g = 0.d0
     return
  end if

  if( (.not. associated(vshx)) ) then
     allocate( vshx(N) )
     allocate( vshw(N) )
     call GaussLegendre__getPoints( N, 0.d0, M_PI, vshx, vshw )
  endif

  tsum=0.d0
  do it=1, N
     th  = vshx(it)
     weight = vshw(it)
     tsum = tsum + weight*sin(th)* &
          SphericalHarmonic__Theta(la,ma,th)* &
          SphericalHarmonic__Theta(lb,mb,th)* &
          SphericalHarmonic__Theta(l,m,th)
  end do

  if( tsum == 0.0 ) then
     g = 0.d0
     return
  endif

  psum=0.d0
  do ip=1, N
     ph     = vshx(ip)*2
     weight = vshw(ip)*2
     psum = psum + weight* &
          SphericalHarmonic__Phi(ma,ph)* &
          SphericalHarmonic__Phi(mb,ph)* &
          SphericalHarmonic__Phi(m,ph)
  end do

  if( psum == 0.0 ) then
     g = 0.d0
     return
  endif

  if( mod(la-lb-l,4) == +2 .or. mod(la-lb-l,4) == -2 ) then
     tsum = -tsum
  endif

  g = tsum*psum

  return
end subroutine SphericalHarmonic__Gaunt

function SphericalHarmonic__Theta(l,m,th) result(Theta)
  use ac_sharmonics

  implicit none
  integer, intent(in) :: l, m
  real(8) :: th
  real(8) :: Theta

  real(8) :: c, s

  c = cos(th)
  s = sin(th)

  Theta = 0.d0

  select case(l)

  case(0)
     select case(abs(m))
     case(0)
        Theta = C00
     end select

  case(1)
     select case(abs(m))
     case(0)
        Theta = C10*c
     case(1)
        Theta = C11*s
     end select

  case(2)
     select case(abs(m))
     case(0)
        Theta = C20*( 3*c*c - 1.0 )
     case(1)
        Theta = C21*( c*s )
     case(2)
        Theta = C22*( s*s )
     end select

  case(3)
     select case(abs(m))
     case(0)
        Theta = C30*c*( 5*c*c - 3.0 )
     case(1)
        Theta = C31*s*( 5*c*c - 1.0 )
     case(2)
        Theta = C32*c*( s*s )
     case(3)
        Theta = C33*s*( s*s )
     end select

  case (4)
     select case(abs(m))
     case(0)
        Theta = C40*( 35*c*c*c*c - 30*c*c + 3 )
     case(1)
        Theta = C41*s*( 7*c*c*c - 3*c )
     case(2)
        Theta = C42*s*s*( 7*c*c - 1 )
     case(3)
        Theta = C43*c*s*s*s
     case(4)
        Theta = C44*s*s*s*s
     end select

  case (5)
     select case(abs(m))
     case(0)
        Theta = C50*c*( 63*c*c*c*c - 70*c*c + 15 )
     case(1)
        Theta = C51*s*( 21*c*c*c*c - 14*c*c + 1 )
     case(2)
        Theta = C52*c*s*s*( 3*c*c - 1 )
     case(3)
        Theta = C53*s*s*s*( 9*c*c - 1 )
     case(4)
        Theta = C54*c*s*s*s*s
     case(5)
        Theta = C55*s*s*s*s*s
     end select

  case(6)
     select case(abs(m))
     case(0)
        Theta = &
             C60*( 231*c*c*c*c*c*c - 315*c*c*c*c + 105*c*c - 5 )
     case(1)
        Theta = &
             C61*c*s*( 33*c*c*c*c - 30*c*c + 5 )
     case(2)
        Theta = &
             C62*s*s*( 33*c*c*c*c - 18*c*c + 1 )
     case(3)
        Theta = &
             C63*c*s*s*s*( 11*c*c - 3 )
     case(4)
        Theta = &
             C64*s*s*s*s*( 11*c*c - 1 )
     case(5)
        Theta = &
             C65*c*s*s*s*s*s
     case(6)
        Theta = &
             C66*s*s*s*s*s*s
     end select

  case default
     Theta = 0.d0
     write(*,'(a70)') '# Sorry : Spherical Harmonic function for l>6 is not implemented.'
     stop

  end select

  return
end function SphericalHarmonic__Theta

function SphericalHarmonic__Phi(m,ph) result(Phi)
  implicit none
  integer       :: m
  real(8) :: ph
  real(8) :: Phi

  if( m == 0 ) then
     Phi = 1.d0
  elseif( m > 0 ) then 
     Phi = cos(m*ph)
  else 
     Phi = sin(m*ph)
  endif

  return
end function SphericalHarmonic__Phi

subroutine SphericalHarmonic__deallocate
  use ac_sharmonics

  implicit none

  if( associated(vshx) ) deallocate(vshx)
  if( associated(vshw) ) deallocate(vshw)

  return
end subroutine SphericalHarmonic__deallocate

